/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */
#ifndef _TRANSCEIVERIMPL_H_
#define _TRANSCEIVERIMPL_H_

#include <string>
#include "libwebrtcglue/MediaConduitControl.h"
#include "mozilla/RefPtr.h"
#include "nsCOMPtr.h"
#include "nsISerialEventTarget.h"
#include "nsTArray.h"
#include "mozilla/dom/MediaStreamTrack.h"
#include "ErrorList.h"
#include "jsep/JsepTransceiver.h"
#include "transport/transportlayer.h"  // For TransportLayer::State

class nsIPrincipal;

namespace mozilla {
class PeerIdentity;
class JsepTransceiver;
class MediaSessionConduit;
class VideoSessionConduit;
class AudioSessionConduit;
struct AudioCodecConfig;
class VideoCodecConfig;  // Why is this a class, but AudioCodecConfig a struct?
class MediaPipelineTransmit;
class MediaPipeline;
class MediaPipelineFilter;
class MediaTransportHandler;
class RTCStatsIdGenerator;
class WebrtcCallWrapper;
class JsepTrackNegotiatedDetails;
class PeerConnectionImpl;

namespace dom {
class RTCDtlsTransport;
class RTCDTMFSender;
class RTCRtpTransceiver;
struct RTCRtpSourceEntry;
class RTCRtpReceiver;
class RTCRtpSender;
}  // namespace dom

/**
 * This is what ties all the various pieces that make up a transceiver
 * together. This includes:
 * MediaStreamTrack for rendering and capture
 * MediaTransportHandler for RTP transmission/reception
 * Audio/VideoConduit for feeding RTP/RTCP into webrtc.org for decoding, and
 * feeding audio/video frames into webrtc.org for encoding into RTP/RTCP.
 */
class TransceiverImpl : public nsISupports,
                        public nsWrapperCache,
                        public sigslot::has_slots<> {
 public:
  /**
   * |aSendTrack| might or might not be set.
   */
  TransceiverImpl(nsPIDOMWindowInner* aWindow, bool aPrivacyNeeded,
                  PeerConnectionImpl* aPc,
                  MediaTransportHandler* aTransportHandler,
                  JsepTransceiver* aJsepTransceiver,
                  nsISerialEventTarget* aStsThread,
                  dom::MediaStreamTrack* aSendTrack,
                  WebrtcCallWrapper* aCallWrapper,
                  RTCStatsIdGenerator* aIdGenerator);

  bool IsValid() const { return !!mConduit; }

  nsresult UpdateTransport();

  nsresult UpdateConduit();

  void ResetSync();

  nsresult SyncWithMatchingVideoConduits(
      nsTArray<RefPtr<TransceiverImpl>>& transceivers);

  void Shutdown_m();

  bool ConduitHasPluginID(uint64_t aPluginID);

  // for webidl
  JSObject* WrapObject(JSContext* aCx,
                       JS::Handle<JSObject*> aGivenProto) override;
  nsPIDOMWindowInner* GetParentObject() const;
  void SyncWithJS(dom::RTCRtpTransceiver& aJsTransceiver, ErrorResult& aRv);
  dom::RTCRtpReceiver* Receiver() const { return mReceiver; }
  dom::RTCRtpSender* Sender() const { return mSender; }
  dom::RTCDtlsTransport* GetDtlsTransport() const { return mDtlsTransport; }
  void GetKind(nsAString& aKind) const;

  bool CanSendDTMF() const;
  bool Stopped() const { return mStopped; }

  void UpdateDtlsTransportState(const std::string& aTransportId,
                                TransportLayer::State aState);
  void SetDtlsTransport(dom::RTCDtlsTransport* aDtlsTransport, bool aStable);
  void RollbackToStableDtlsTransport();

  std::string GetTransportId() const {
    return mJsepTransceiver->mTransport.mTransportId;
  }

  bool IsVideo() const;

  bool IsSending() const {
    return !mJsepTransceiver->IsStopped() &&
           mJsepTransceiver->mSendTrack.GetActive();
  }

  bool IsReceiving() const {
    return !mJsepTransceiver->IsStopped() &&
           mJsepTransceiver->mRecvTrack.GetActive();
  }

  Maybe<const std::vector<UniquePtr<JsepCodecDescription>>&>
  GetNegotiatedSendCodecs() const;

  Maybe<const std::vector<UniquePtr<JsepCodecDescription>>&>
  GetNegotiatedRecvCodecs() const;

  struct PayloadTypes {
    Maybe<int> mSendPayloadType;
    Maybe<int> mRecvPayloadType;
  };
  using ActivePayloadTypesPromise = MozPromise<PayloadTypes, nsresult, true>;
  RefPtr<ActivePayloadTypesPromise> GetActivePayloadTypes() const;

  MediaSessionConduit* GetConduit() const { return mConduit; }

  // nsISupports
  NS_DECL_CYCLE_COLLECTING_ISUPPORTS
  NS_DECL_CYCLE_COLLECTION_SCRIPT_HOLDER_CLASS(TransceiverImpl)

  static void NegotiatedDetailsToAudioCodecConfigs(
      const JsepTrackNegotiatedDetails& aDetails,
      std::vector<AudioCodecConfig>* aConfigs);

  static void NegotiatedDetailsToVideoCodecConfigs(
      const JsepTrackNegotiatedDetails& aDetails,
      std::vector<VideoCodecConfig>* aConfigs);

  /* Returns a promise that will contain the stats in aStats, along with the
   * codec stats (which is a PC-wide thing) */
  void ChainToDomPromiseWithCodecStats(
      nsTArray<RefPtr<dom::RTCStatsPromise>> aStats,
      const RefPtr<dom::Promise>& aDomPromise);

  /**
   * Takes a set of codec stats (per-peerconnection) and a set of
   * transceiver/transceiver-stats-promise tuples. Filters out all referenced
   * codec stats based on the transceiver's transport and rtp stream stats.
   * Finally returns the flattened stats containing the filtered codec stats and
   * all given per-transceiver-stats.
   */
  static RefPtr<dom::RTCStatsPromise> ApplyCodecStats(
      nsTArray<dom::RTCCodecStats> aCodecStats,
      nsTArray<std::tuple<TransceiverImpl*,
                          RefPtr<dom::RTCStatsPromise::AllPromiseType>>>
          aTransceiverStatsPromises);

  AbstractCanonical<std::string>* CanonicalMid() { return &mMid; }
  AbstractCanonical<std::string>* CanonicalSyncGroup() { return &mSyncGroup; }

 private:
  virtual ~TransceiverImpl();
  void InitAudio();
  void InitVideo();
  void InitConduitControl();
  void Stop();

  nsCOMPtr<nsPIDOMWindowInner> mWindow;
  RefPtr<PeerConnectionImpl> mPc;
  RefPtr<MediaTransportHandler> mTransportHandler;
  const RefPtr<JsepTransceiver> mJsepTransceiver;
  nsCOMPtr<nsISerialEventTarget> mStsThread;
  // state for webrtc.org that is shared between all transceivers
  RefPtr<WebrtcCallWrapper> mCallWrapper;
  RefPtr<RTCStatsIdGenerator> mIdGenerator;
  RefPtr<MediaSessionConduit> mConduit;
  // The spec says both RTCRtpReceiver and RTCRtpSender have a slot for
  // an RTCDtlsTransport.  They are always the same, so we'll store it
  // here.
  RefPtr<dom::RTCDtlsTransport> mDtlsTransport;
  // The spec says both RTCRtpReceiver and RTCRtpSender have a slot for
  // a last stable state RTCDtlsTransport.  They are always the same, so
  // we'll store it here.
  RefPtr<dom::RTCDtlsTransport> mLastStableDtlsTransport;
  RefPtr<dom::RTCRtpReceiver> mReceiver;
  RefPtr<dom::RTCRtpSender> mSender;
  bool mStopped = false;
  bool mShutdown = false;
  bool mSyncing = false;

  Canonical<std::string> mMid;
  Canonical<std::string> mSyncGroup;
};

}  // namespace mozilla

#endif  // _TRANSCEIVERIMPL_H_
