/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

this.EXPORTED_SYMBOLS = ["TestRunner"];

const {classes: Cc, interfaces: Ci, utils: Cu} = Components;
const env = Cc["@mozilla.org/process/environment;1"].getService(Ci.nsIEnvironment);
const APPLY_CONFIG_TIMEOUT_MS = 60 * 1000;
const HOME_PAGE = "chrome://mozscreenshots/content/lib/mozscreenshots.html";

Cu.import("resource://gre/modules/FileUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/Timer.jsm");
Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/osfile.jsm");
Cu.import("resource://gre/modules/Geometry.jsm");

XPCOMUtils.defineLazyModuleGetter(this, "BrowserTestUtils",
                                  "resource://testing-common/BrowserTestUtils.jsm");

XPCOMUtils.defineLazyModuleGetter(this, "Screenshot", "chrome://mozscreenshots/content/Screenshot.jsm");

// Create a new instance of the ConsoleAPI so we can control the maxLogLevel with a pref.
// See LOG_LEVELS in Console.jsm. Common examples: "All", "Info", "Warn", & "Error".
const PREF_LOG_LEVEL = "extensions.mozscreenshots@mozilla.org.loglevel";
XPCOMUtils.defineLazyGetter(this, "log", () => {
  let ConsoleAPI = Cu.import("resource://gre/modules/Console.jsm", {}).ConsoleAPI;
  let consoleOptions = {
    maxLogLevel: "info",
    maxLogLevelPref: PREF_LOG_LEVEL,
    prefix: "mozscreenshots",
  };
  return new ConsoleAPI(consoleOptions);
});

this.TestRunner = {
  combos: null,
  completedCombos: 0,
  currentComboIndex: 0,
  _lastCombo: null,
  _libDir: null,
  croppingPadding: 10,

  init(extensionPath) {
    log.debug("init");
    this._extensionPath = extensionPath;
  },

  /**
   * Load specified sets, execute all combinations of them, and capture screenshots.
   */
  async start(setNames, jobName = null) {
    let subDirs = ["mozscreenshots",
                   (new Date()).toISOString().replace(/:/g, "-") + "_" + Services.appinfo.OS];
    let screenshotPath = FileUtils.getFile("TmpD", subDirs).path;

    const MOZ_UPLOAD_DIR = env.get("MOZ_UPLOAD_DIR");
    if (MOZ_UPLOAD_DIR) {
      screenshotPath = MOZ_UPLOAD_DIR;
    }

    log.info("Saving screenshots to:", screenshotPath);

    let screenshotPrefix = Services.appinfo.appBuildID;
    if (jobName) {
      screenshotPrefix += "-" + jobName;
    }
    screenshotPrefix += "_";
    Screenshot.init(screenshotPath, this._extensionPath, screenshotPrefix);
    this._libDir = this._extensionPath.QueryInterface(Ci.nsIFileURL).file.clone();
    this._libDir.append("chrome");
    this._libDir.append("mozscreenshots");
    this._libDir.append("lib");

    let sets = this.loadSets(setNames);

    log.info(sets.length + " sets:", setNames);
    this.combos = new LazyProduct(sets);
    log.info(this.combos.length + " combinations");

    this.currentComboIndex = this.completedCombos = 0;
    this._lastCombo = null;

    // Setup some prefs
    Services.prefs.setCharPref("browser.aboutHomeSnippets.updateUrl",
                               "data:text/html;charset=utf-8,Generated by mozscreenshots");
    Services.prefs.setCharPref("extensions.ui.lastCategory", "addons://list/extension");
    // Don't let the caret blink since it causes false positives for image diffs
    Services.prefs.setIntPref("ui.caretBlinkTime", -1);
    // Disable some animations that can cause false positives, such as the
    // reload/stop button spinning animation.
    Services.prefs.setBoolPref("toolkit.cosmeticAnimations.enabled", false);

    let browserWindow = Services.wm.getMostRecentWindow("navigator:browser");

    // When being automated through Marionette, Firefox shows a prominent indication
    // in the urlbar and identity block. We don't want this to show when testing browser UI.
    // Note that this doesn't prevent subsequently opened windows from showing the automation UI.
    browserWindow.document.getElementById("main-window").removeAttribute("remotecontrol");

    let selectedBrowser = browserWindow.gBrowser.selectedBrowser;
    await BrowserTestUtils.loadURI(selectedBrowser, HOME_PAGE);
    await BrowserTestUtils.browserLoaded(selectedBrowser);

    for (let i = 0; i < this.combos.length; i++) {
      this.currentComboIndex = i;
      await this._performCombo(this.combos.item(this.currentComboIndex));
    }

    log.info("Done: Completed " + this.completedCombos + " out of " +
             this.combos.length + " configurations.");
    this.cleanup();
  },

  /**
   * Helper function for loadSets. This filters out the restricted configs from setName.
   * This was made a helper function to facilitate xpcshell unit testing.
   * @param {String} setName - set name to be filtered e.g. "Toolbars[onlyNavBar,allToolbars]"
   * @return {Object} Returns an object with two values: the filtered set name and a set of
   *                  restricted configs.
   */
  filterRestrictions(setName) {
    let match = /\[([^\]]+)\]$/.exec(setName);
    if (!match) {
      throw new Error(`Invalid restrictions in ${setName}`);
    }
    // Trim the restrictions from the set name.
    setName = setName.slice(0, match.index);
    let restrictions = match[1].split(",").reduce((set, name) => set.add(name.trim())
                                                 , new Set());

    return { trimmedSetName: setName, restrictions };
  },

  /**
   * Load sets of configurations from JSMs.
   * @param {String[]} setNames - array of set names (e.g. ["Tabs", "WindowSize"].
   * @return {Object[]} Array of sets containing `name` and `configurations` properties.
   */
  loadSets(setNames) {
    let sets = [];
    for (let setName of setNames) {
      let restrictions = null;
      if (setName.includes("[")) {
        let filteredData = this.filterRestrictions(setName);
        setName = filteredData.trimmedSetName;
        restrictions = filteredData.restrictions;
      }
      try {
        let imported = {};
        Cu.import("chrome://mozscreenshots/content/configurations/" + setName + ".jsm",
                  imported);
        imported[setName].init(this._libDir);
        let configurationNames = Object.keys(imported[setName].configurations);
        if (!configurationNames.length) {
          throw new Error(setName + " has no configurations for this environment");
        }
        // Checks to see if nonexistent configuration have been specified
        if (restrictions) {
          let incorrectConfigs = [...restrictions].filter(r => !configurationNames.includes(r));
          if (incorrectConfigs.length) {
            throw new Error("non existent configurations: " + incorrectConfigs);
          }
        }
        let configurations = {};
        for (let config of configurationNames) {
          // Automatically set the name property of the configuration object to
          // its name from the configuration object.
          imported[setName].configurations[config].name = config;
          // Filter restricted configurations.
          if (!restrictions || restrictions.has(config)) {
            configurations[config] = imported[setName].configurations[config];
          }
        }
        sets.push(configurations);
      } catch (ex) {
        log.error("Error loading set: " + setName);
        log.error(ex);
        throw ex;
      }
    }
    return sets;
  },

  cleanup() {
    let browserWindow = Services.wm.getMostRecentWindow("navigator:browser");
    let gBrowser = browserWindow.gBrowser;
    while (gBrowser.tabs.length > 1) {
      gBrowser.removeTab(gBrowser.selectedTab, {animate: false});
    }
    gBrowser.unpinTab(gBrowser.selectedTab);
    gBrowser.selectedBrowser.loadURI("data:text/html;charset=utf-8,<h1>Done!");
    browserWindow.restore();
    Services.prefs.clearUserPref("toolkit.cosmeticAnimations.enabled");
  },

  // helpers

  /**
  * Calculate the bounding box based on CSS selector from config for cropping
  *
  * @param {String[]} selectors - array of CSS selectors for relevant DOM element
  * @return {Geometry.jsm Rect} Rect holding relevant x, y, width, height with padding
  **/
  _findBoundingBox(selectors, windowType) {
    // No selectors provided
    if (!selectors.length) {
      log.info("_findBoundingBox: selectors argument is empty");
      return null;
    }

    // Set window type, default "navigator:browser"
    windowType = windowType || "navigator:browser";
    let browserWindow = Services.wm.getMostRecentWindow(windowType);
    // Scale for high-density displays
    const scale = browserWindow.QueryInterface(Ci.nsIInterfaceRequestor)
                        .getInterface(Ci.nsIDocShell).QueryInterface(Ci.nsIBaseWindow)
                        .devicePixelsPerDesktopPixel;

    let finalRect = undefined;
    // Grab bounding boxes and find the union
    for (let selector of selectors) {
      let element;
      // Check for function to find anonymous content
      if (typeof(selector) == "function") {
        element = selector();
      } else {
        element = browserWindow.document.querySelector(selector);
      }

      // Selector not found
      if (!element) {
        log.info("_findBoundingBox: selector not found");
        return null;
      }

      // Calculate box region, convert to Rect
      let box = element.ownerDocument.getBoxObjectFor(element);
      let newRect = new Rect(box.screenX * scale, box.screenY * scale,
                             box.width * scale, box.height * scale);

      if (!finalRect) {
        finalRect = newRect;
      } else {
        finalRect = finalRect.union(newRect);
      }
    }

    // Add fixed padding
    finalRect = finalRect.inflateFixed(this.croppingPadding * scale);

    let windowLeft = browserWindow.screenX * scale;
    let windowTop = browserWindow.screenY * scale;
    let windowWidth = browserWindow.outerWidth * scale;
    let windowHeight = browserWindow.outerHeight * scale;

    // Clip dimensions to window only
    finalRect.left = Math.max(finalRect.left, windowLeft);
    finalRect.top = Math.max(finalRect.top, windowTop);
    finalRect.right = Math.min(finalRect.right, windowLeft + windowWidth);
    finalRect.bottom = Math.min(finalRect.bottom, windowTop + windowHeight);

    return finalRect;
  },

  async _performCombo(combo) {
    let paddedComboIndex = padLeft(this.currentComboIndex + 1, String(this.combos.length).length);
    log.info("Combination " + paddedComboIndex + "/" + this.combos.length + ": " +
             this._comboName(combo).substring(1));

    function changeConfig(config) {
      log.debug("calling " + config.name);
      let applyPromise = Promise.resolve(config.applyConfig());
      let timeoutPromise = new Promise((resolve, reject) => {
        setTimeout(reject, APPLY_CONFIG_TIMEOUT_MS, "Timed out");
      });
      log.debug("called " + config.name);
      // Add a default timeout of 500ms to avoid conflicts when configurations
      // try to apply at the same time. e.g WindowSize and TabsInTitlebar
      return Promise.race([applyPromise, timeoutPromise]).then(() => {
        return new Promise((resolve) => {
          setTimeout(resolve, 500);
        });
      });
    }

    try {
      // First go through and actually apply all of the configs
      for (let i = 0; i < combo.length; i++) {
        let config = combo[i];
        if (!this._lastCombo || config !== this._lastCombo[i]) {
          log.debug("promising", config.name);
          await changeConfig(config);
        }
      }

      // Update the lastCombo since it's now been applied regardless of whether it's accepted below.
      log.debug("fulfilled all applyConfig so setting lastCombo.");
      this._lastCombo = combo;

      // Then ask configs if the current setup is valid. We can't can do this in
      // the applyConfig methods of the config since it doesn't know what configs
      // later in the loop will do that may invalidate the combo.
      for (let i = 0; i < combo.length; i++) {
        let config = combo[i];
        // A configuration can specify an optional verifyConfig method to indicate
        // if the current config is valid for a screenshot. This gets called even
        // if the this config was used in the lastCombo since another config may
        // have invalidated it.
        if (config.verifyConfig) {
          log.debug("checking if the combo is valid with", config.name);
          await config.verifyConfig();
        }
      }
    } catch (ex) {
      log.warn("\tskipped configuration: " + ex);
      // Don't set lastCombo here so that we properly know which configurations
      // need to be applied since the last screenshot

      // Return so we don't take a screenshot.
      return;
    }

    await this._onConfigurationReady(combo);
  },

  _onConfigurationReady(combo) {
    let delayedScreenshot = () => {
      let filename = padLeft(this.currentComboIndex + 1,
                             String(this.combos.length).length) + this._comboName(combo);
      return Screenshot.captureExternal(filename)
        .then(() => {
          this.completedCombos++;
        });
    };

    log.debug("_onConfigurationReady");
    return delayedScreenshot();
  },

  _comboName(combo) {
    return combo.reduce(function(a, b) {
      return a + "_" + b.name;
    }, "");
  },

  /**
   * Finds the index of the first comma that is not enclosed within square brackets.
   * @param {String} envVar - the string that needs to be searched
   * @return {Integer} index of valid comma or -1 if not found.
   */
  findComma(envVar) {
    let nestingDepth = 0;
    for (let i = 0; i < envVar.length; i++) {
      if (envVar[i] === "[") {
        nestingDepth += 1;
      } else if (envVar[i] === "]") {
        nestingDepth -= 1;
      } else if (envVar[i] === "," && nestingDepth === 0) {
        return i;
      }
    }

    return -1;
  },

  /**
   * Splits the environment variable around commas not enclosed in brackets.
   * @param {String} envVar - The environment variable
   * @return {String[]} Array of strings containing the configurations
   * e.g. ["Toolbars[onlyNavBar,allToolbars]","DevTools[jsdebugger,webconsole]","Tabs"]
   */
  splitEnv(envVar) {
    let result = [];

    let commaIndex = this.findComma(envVar);
    while (commaIndex != -1) {
      result.push(envVar.slice(0, commaIndex).trim());
      envVar = envVar.slice(commaIndex + 1);
      commaIndex = this.findComma(envVar);
    }
    result.push(envVar.trim());
    return result;
  }
};

/**
 * Helper to lazily compute the Cartesian product of all of the sets of configurations.
 **/
function LazyProduct(sets) {
  /**
   * An entry for each set with the value being:
   * [the number of permutations of the sets with lower index,
   *  the number of items in the set at the index]
   */
  this.sets = sets;
  this.lookupTable = [];
  let combinations = 1;
  for (let i = this.sets.length - 1; i >= 0; i--) {
    let set = this.sets[i];
    let setLength = Object.keys(set).length;
    this.lookupTable[i] = [combinations, setLength];
    combinations *= setLength;
  }
}
LazyProduct.prototype = {
  get length() {
    let last = this.lookupTable[0];
    if (!last)
      return 0;
    return last[0] * last[1];
  },

  item(n) {
    // For set i, get the item from the set with the floored value of
    // (n / the number of permutations of the sets already chosen from) modulo the length of set i
    let result = [];
    for (let i = this.sets.length - 1; i >= 0; i--) {
      let priorCombinations = this.lookupTable[i][0];
      let setLength = this.lookupTable[i][1];
      let keyIndex = Math.floor(n / priorCombinations) % setLength;
      let keys = Object.keys(this.sets[i]);
      result[i] = this.sets[i][keys[keyIndex]];
    }
    return result;
  },
};

function padLeft(number, width, padding = "0") {
  return padding.repeat(Math.max(0, width - String(number).length)) + number;
}
