/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
"use strict";

this.EXPORTED_SYMBOLS = ["WebVTT"];

/**
 * Code below is vtt.js the JS WebVTT implementation.
 * Current source code can be found at http://github.com/mozilla/vtt.js
 *
 * Code taken from commit b89bfd06cd788a68c67e03f44561afe833db0849
 */
/**
 * Copyright 2013 vtt.js Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var Cu = Components.utils;
Cu.import('resource://gre/modules/Services.jsm');
const { require } = Cu.import("resource://devtools/shared/Loader.jsm", {});
const { XPCOMUtils } = require("resource://gre/modules/XPCOMUtils.jsm");

(function(global) {

  var _objCreate = Object.create || (function() {
    function F() {}
    return function(o) {
      if (arguments.length !== 1) {
        throw new Error('Object.create shim only accepts one parameter.');
      }
      F.prototype = o;
      return new F();
    };
  })();

  // Creates a new ParserError object from an errorData object. The errorData
  // object should have default code and message properties. The default message
  // property can be overriden by passing in a message parameter.
  // See ParsingError.Errors below for acceptable errors.
  function ParsingError(errorData, message) {
    this.name = "ParsingError";
    this.code = errorData.code;
    this.message = message || errorData.message;
  }
  ParsingError.prototype = _objCreate(Error.prototype);
  ParsingError.prototype.constructor = ParsingError;

  // ParsingError metadata for acceptable ParsingErrors.
  ParsingError.Errors = {
    BadSignature: {
      code: 0,
      message: "Malformed WebVTT signature."
    },
    BadTimeStamp: {
      code: 1,
      message: "Malformed time stamp."
    }
  };

  // See spec, https://w3c.github.io/webvtt/#collect-a-webvtt-timestamp.
  function collectTimeStamp(input) {
    function computeSeconds(h, m, s, f) {
      if (m > 59 || s > 59) {
        return null;
      }
      // The attribute of the milli-seconds can only be three digits.
      if (f.length !== 3) {
        return null;
      }
      return (h | 0) * 3600 + (m | 0) * 60 + (s | 0) + (f | 0) / 1000;
    }

    var timestamp = input.match(/^(\d+:)?(\d{2}):(\d{2})\.(\d+)/);
    if (!timestamp || timestamp.length !== 5) {
      return null;
    }

    let hours = timestamp[1]? timestamp[1].replace(":", "") : 0;
    let minutes = timestamp[2];
    let seconds = timestamp[3];
    let milliSeconds = timestamp[4];

    return computeSeconds(hours, minutes, seconds, milliSeconds);
  }

  // A settings object holds key/value pairs and will ignore anything but the first
  // assignment to a specific key.
  function Settings() {
    this.values = _objCreate(null);
  }

  Settings.prototype = {
    set: function(k, v) {
      if (v !== "") {
        this.values[k] = v;
      }
    },
    // Return the value for a key, or a default value.
    // If 'defaultKey' is passed then 'dflt' is assumed to be an object with
    // a number of possible default values as properties where 'defaultKey' is
    // the key of the property that will be chosen; otherwise it's assumed to be
    // a single value.
    get: function(k, dflt, defaultKey) {
      if (defaultKey) {
        return this.has(k) ? this.values[k] : dflt[defaultKey];
      }
      return this.has(k) ? this.values[k] : dflt;
    },
    // Check whether we have a value for a key.
    has: function(k) {
      return k in this.values;
    },
    // Accept a setting if its one of the given alternatives.
    alt: function(k, v, a) {
      for (var n = 0; n < a.length; ++n) {
        if (v === a[n]) {
          this.set(k, v);
          break;
        }
      }
    },
    // Accept a setting if its a valid digits value (int or float)
    digitsValue: function(k, v) {
      if (/^-0+(\.[0]*)?$/.test(v)) { // special case for -0.0
        this.set(k, 0.0);
      } else if (/^-?\d+(\.[\d]*)?$/.test(v)) {
        this.set(k, parseFloat(v));
      }
    },
    // Accept a setting if its a valid percentage.
    percent: function(k, v) {
      var m;
      if ((m = v.match(/^([\d]{1,3})(\.[\d]*)?%$/))) {
        v = parseFloat(v);
        if (v >= 0 && v <= 100) {
          this.set(k, v);
          return true;
        }
      }
      return false;
    }
  };

  // Helper function to parse input into groups separated by 'groupDelim', and
  // interprete each group as a key/value pair separated by 'keyValueDelim'.
  function parseOptions(input, callback, keyValueDelim, groupDelim) {
    var groups = groupDelim ? input.split(groupDelim) : [input];
    for (var i in groups) {
      if (typeof groups[i] !== "string") {
        continue;
      }
      var kv = groups[i].split(keyValueDelim);
      if (kv.length !== 2) {
        continue;
      }
      var k = kv[0];
      var v = kv[1];
      callback(k, v);
    }
  }

  function parseCue(input, cue, regionList) {
    // Remember the original input if we need to throw an error.
    var oInput = input;
    // 4.1 WebVTT timestamp
    function consumeTimeStamp() {
      var ts = collectTimeStamp(input);
      if (ts === null) {
        throw new ParsingError(ParsingError.Errors.BadTimeStamp,
                              "Malformed timestamp: " + oInput);
      }
      // Remove time stamp from input.
      input = input.replace(/^[^\s\uFFFDa-zA-Z-]+/, "");
      return ts;
    }

    // 4.4.2 WebVTT cue settings
    function consumeCueSettings(input, cue) {
      var settings = new Settings();

      parseOptions(input, function (k, v) {
        switch (k) {
        case "region":
          // Find the last region we parsed with the same region id.
          for (var i = regionList.length - 1; i >= 0; i--) {
            if (regionList[i].id === v) {
              settings.set(k, regionList[i].region);
              break;
            }
          }
          break;
        case "vertical":
          settings.alt(k, v, ["rl", "lr"]);
          break;
        case "line":
          var vals = v.split(","),
              vals0 = vals[0];
          settings.digitsValue(k, vals0);
          settings.percent(k, vals0) ? settings.set("snapToLines", false) : null;
          settings.alt(k, vals0, ["auto"]);
          if (vals.length === 2) {
            settings.alt("lineAlign", vals[1], ["start", "center", "end"]);
          }
          break;
        case "position":
          vals = v.split(",");
          settings.percent(k, vals[0]);
          if (vals.length === 2) {
            settings.alt("positionAlign", vals[1], ["line-left", "center", "line-right", "auto"]);
          }
          break;
        case "size":
          settings.percent(k, v);
          break;
        case "align":
          settings.alt(k, v, ["start", "center", "end", "left", "right"]);
          break;
        }
      }, /:/, /\s/);

      // Apply default values for any missing fields.
      cue.region = settings.get("region", null);
      cue.vertical = settings.get("vertical", "");
      cue.line = settings.get("line", "auto");
      cue.lineAlign = settings.get("lineAlign", "start");
      cue.snapToLines = settings.get("snapToLines", true);
      cue.size = settings.get("size", 100);
      cue.align = settings.get("align", "center");
      cue.position = settings.get("position", "auto");
      cue.positionAlign = settings.get("positionAlign", "center");
    }

    function skipWhitespace() {
      input = input.replace(/^[ \f\n\r\t]+/, "");
    }

    // 4.1 WebVTT cue timings.
    skipWhitespace();
    cue.startTime = consumeTimeStamp();   // (1) collect cue start time
    skipWhitespace();
    if (input.substr(0, 3) !== "-->") {     // (3) next characters must match "-->"
      throw new ParsingError(ParsingError.Errors.BadTimeStamp,
                             "Malformed time stamp (time stamps must be separated by '-->'): " +
                             oInput);
    }
    input = input.substr(3);
    skipWhitespace();
    cue.endTime = consumeTimeStamp();     // (5) collect cue end time

    // 4.1 WebVTT cue settings list.
    skipWhitespace();
    consumeCueSettings(input, cue);
  }

  function onlyContainsWhiteSpaces(input) {
    return /^[ \f\n\r\t]+$/.test(input);
  }

  function containsTimeDirectionSymbol(input) {
    return input.indexOf("-->") !== -1;
  }

  function maybeIsTimeStampFormat(input) {
    return /^\s*(\d+:)?(\d{2}):(\d{2})\.(\d+)\s*-->\s*(\d+:)?(\d{2}):(\d{2})\.(\d+)\s*/.test(input);
  }

  var ESCAPE = {
    "&amp;": "&",
    "&lt;": "<",
    "&gt;": ">",
    "&lrm;": "\u200e",
    "&rlm;": "\u200f",
    "&nbsp;": "\u00a0"
  };

  var TAG_NAME = {
    c: "span",
    i: "i",
    b: "b",
    u: "u",
    ruby: "ruby",
    rt: "rt",
    v: "span",
    lang: "span"
  };

  var TAG_ANNOTATION = {
    v: "title",
    lang: "lang"
  };

  var NEEDS_PARENT = {
    rt: "ruby"
  };

  const PARSE_CONTENT_MODE = {
    NORMAL_CUE: "normal_cue",
    PSUEDO_CUE: "pseudo_cue",
    DOCUMENT_FRAGMENT: "document_fragment",
    REGION_CUE: "region_cue",
  }
  // Parse content into a document fragment.
  function parseContent(window, input, mode) {
    function nextToken() {
      // Check for end-of-string.
      if (!input) {
        return null;
      }

      // Consume 'n' characters from the input.
      function consume(result) {
        input = input.substr(result.length);
        return result;
      }

      var m = input.match(/^([^<]*)(<[^>]+>?)?/);
      // The input doesn't contain a complete tag.
      if (!m[0]) {
        return null;
      }
      // If there is some text before the next tag, return it, otherwise return
      // the tag.
      return consume(m[1] ? m[1] : m[2]);
    }

    // Unescape a string 's'.
    function unescape1(e) {
      return ESCAPE[e];
    }
    function unescape(s) {
      while ((m = s.match(/&(amp|lt|gt|lrm|rlm|nbsp);/))) {
        s = s.replace(m[0], unescape1);
      }
      return s;
    }

    function shouldAdd(current, element) {
      return !NEEDS_PARENT[element.localName] ||
             NEEDS_PARENT[element.localName] === current.localName;
    }

    // Create an element for this tag.
    function createElement(type, annotation) {
      var tagName = TAG_NAME[type];
      if (!tagName) {
        return null;
      }
      var element = window.document.createElement(tagName);
      var name = TAG_ANNOTATION[type];
      if (name) {
        element[name] = annotation ? annotation.trim() : "";
      }
      return element;
    }

    // https://w3c.github.io/webvtt/#webvtt-timestamp-object
    // Return hhhhh:mm:ss.fff
    function normalizedTimeStamp(secondsWithFrag) {
      var totalsec = parseInt(secondsWithFrag, 10);
      var hours = Math.floor(totalsec / 3600);
      var minutes = Math.floor(totalsec % 3600 / 60);
      var seconds = Math.floor(totalsec % 60);
      if (hours < 10) {
        hours = "0" + hours;
      }
      if (minutes < 10) {
        minutes = "0" + minutes;
      }
      if (seconds < 10) {
        seconds = "0" + seconds;
      }
      var f = secondsWithFrag.toString().split(".");
      if (f[1]) {
        f = f[1].slice(0, 3).padEnd(3, "0");
      } else {
        f = "000";
      }
      return hours + ':' + minutes + ':' + seconds + '.' + f;
    }

    var root;
    switch (mode) {
      case PARSE_CONTENT_MODE.PSUEDO_CUE:
        root = window.document.createElement("div", {pseudo: "::cue"});
        break;
      case PARSE_CONTENT_MODE.NORMAL_CUE:
      case PARSE_CONTENT_MODE.REGION_CUE:
        root = window.document.createElement("div");
        break;
      case PARSE_CONTENT_MODE.DOCUMENT_FRAGMENT:
        root = window.document.createDocumentFragment();
        break;
    }

    var current = root,
        t,
        tagStack = [];

    while ((t = nextToken()) !== null) {
      if (t[0] === '<') {
        if (t[1] === "/") {
          // If the closing tag matches, move back up to the parent node.
          if (tagStack.length &&
              tagStack[tagStack.length - 1] === t.substr(2).replace(">", "")) {
            tagStack.pop();
            current = current.parentNode;
          }
          // Otherwise just ignore the end tag.
          continue;
        }
        var ts = collectTimeStamp(t.substr(1, t.length - 1));
        var node;
        if (ts) {
          // Timestamps are lead nodes as well.
          node = window.document.createProcessingInstruction("timestamp", normalizedTimeStamp(ts));
          current.appendChild(node);
          continue;
        }
        var m = t.match(/^<([^.\s/0-9>]+)(\.[^\s\\>]+)?([^>\\]+)?(\\?)>?$/);
        // If we can't parse the tag, skip to the next tag.
        if (!m) {
          continue;
        }
        // Try to construct an element, and ignore the tag if we couldn't.
        node = createElement(m[1], m[3]);
        if (!node) {
          continue;
        }
        // Determine if the tag should be added based on the context of where it
        // is placed in the cuetext.
        if (!shouldAdd(current, node)) {
          continue;
        }
        // Set the class list (as a list of classes, separated by space).
        if (m[2]) {
          node.className = m[2].substr(1).replace('.', ' ');
        }
        // Append the node to the current node, and enter the scope of the new
        // node.
        tagStack.push(m[1]);
        current.appendChild(node);
        current = node;
        continue;
      }

      // Text nodes are leaf nodes.
      current.appendChild(window.document.createTextNode(unescape(t)));
    }

    return root;
  }

  function StyleBox() {
  }

  // Apply styles to a div. If there is no div passed then it defaults to the
  // div on 'this'.
  StyleBox.prototype.applyStyles = function(styles, div) {
    div = div || this.div;
    for (var prop in styles) {
      if (styles.hasOwnProperty(prop)) {
        div.style[prop] = styles[prop];
      }
    }
  };

  StyleBox.prototype.formatStyle = function(val, unit) {
    return val === 0 ? 0 : val + unit;
  };

  XPCOMUtils.defineLazyPreferenceGetter(StyleBox.prototype, "supportPseudo",
                                        "media.webvtt.pseudo.enabled", false);

  // Constructs the computed display state of the cue (a div). Places the div
  // into the overlay which should be a block level element (usually a div).
  function CueStyleBox(window, cue, styleOptions) {
    var isIE8 = (typeof navigator !== "undefined") &&
      (/MSIE\s8\.0/).test(navigator.userAgent);

    var isFirefoxSupportPseudo = (/firefox/i.test(window.navigator.userAgent))
          && this.supportPseudo;
    var color = "rgba(255, 255, 255, 1)";
    var backgroundColor = "rgba(0, 0, 0, 0.8)";

    if (isIE8) {
      color = "rgb(255, 255, 255)";
      backgroundColor = "rgb(0, 0, 0)";
    }

    StyleBox.call(this);
    this.cue = cue;

    // Parse our cue's text into a DOM tree rooted at 'cueDiv'. This div will
    // have inline positioning and will function as the cue background box.
    if (isFirefoxSupportPseudo) {
      this.cueDiv = parseContent(window, cue.text, PARSE_CONTENT_MODE.PSUEDO_CUE);
    } else {
      this.cueDiv = parseContent(window, cue.text, PARSE_CONTENT_MODE.NORMAL_CUE);
    }
    var styles = {
      color: color,
      backgroundColor: backgroundColor,
      display: "inline",
      font: styleOptions.font,
      whiteSpace: "pre-line",
    };
    if (isFirefoxSupportPseudo) {
      delete styles.color;
      delete styles.backgroundColor;
      delete styles.font;
      delete styles.whiteSpace;
    }

    if (!isIE8) {
      styles.writingMode = cue.vertical === "" ? "horizontal-tb"
                                               : cue.vertical === "lr" ? "vertical-lr"
                                                                       : "vertical-rl";
      styles.unicodeBidi = "plaintext";
    }
    this.applyStyles(styles, this.cueDiv);

    // Create an absolutely positioned div that will be used to position the cue
    // div.
    styles = {
      position: "absolute",
      textAlign: cue.align,
      font: styleOptions.font,
    };

    this.div = window.document.createElement("div");
    this.applyStyles(styles);

    this.div.appendChild(this.cueDiv);

    // Calculate the distance from the reference edge of the viewport to the text
    // position of the cue box. The reference edge will be resolved later when
    // the box orientation styles are applied.
    function convertCuePostionToPercentage(cuePosition) {
      if (cuePosition === "auto") {
        return 50;
      }
      return cuePosition;
    }
    var textPos = 0;
    let postionPercentage = convertCuePostionToPercentage(cue.position);
    switch (cue.computedPositionAlign) {
      // TODO : modify these fomula to follow the spec, see bug 1277437.
      case "line-left":
        textPos = postionPercentage;
        break;
      case "center":
        textPos = postionPercentage - (cue.size / 2);
        break;
      case "line-right":
        textPos = postionPercentage - cue.size;
        break;
    }

    // Horizontal box orientation; textPos is the distance from the left edge of the
    // area to the left edge of the box and cue.size is the distance extending to
    // the right from there.
    if (cue.vertical === "") {
      this.applyStyles({
        left:  this.formatStyle(textPos, "%"),
        width: this.formatStyle(cue.size, "%")
      });
    // Vertical box orientation; textPos is the distance from the top edge of the
    // area to the top edge of the box and cue.size is the height extending
    // downwards from there.
    } else {
      this.applyStyles({
        top: this.formatStyle(textPos, "%"),
        height: this.formatStyle(cue.size, "%")
      });
    }

    this.move = function(box) {
      this.applyStyles({
        top: this.formatStyle(box.top, "px"),
        bottom: this.formatStyle(box.bottom, "px"),
        left: this.formatStyle(box.left, "px"),
        right: this.formatStyle(box.right, "px"),
        height: this.formatStyle(box.height, "px"),
        width: this.formatStyle(box.width, "px")
      });
    };
  }
  CueStyleBox.prototype = _objCreate(StyleBox.prototype);
  CueStyleBox.prototype.constructor = CueStyleBox;

  function RegionNodeBox(window, region, container) {
    StyleBox.call(this);

    var boxLineHeight = container.height * 0.0533 // 0.0533vh ? 5.33vh
    var boxHeight = boxLineHeight * region.lines;
    var boxWidth = container.width * region.width / 100; // convert percentage to px

    var regionNodeStyles = {
      position: "absolute",
      height: boxHeight + "px",
      width: boxWidth + "px",
      top: (region.viewportAnchorY * container.height / 100) - (region.regionAnchorY * boxHeight / 100) + "px",
      left: (region.viewportAnchorX * container.width / 100) - (region.regionAnchorX * boxWidth / 100) + "px",
      lineHeight: boxLineHeight + "px",
      writingMode: "horizontal-tb",
      backgroundColor: "rgba(0, 0, 0, 0.8)",
      wordWrap: "break-word",
      overflowWrap: "break-word",
      font: (boxLineHeight/1.3) + "px sans-serif",
      color: "rgba(255, 255, 255, 1)",
      overflow: "hidden",
      minHeight: "0px",
      maxHeight: boxHeight + "px",
      display: "inline-flex",
      flexFlow: "column",
      justifyContent: "flex-end",
    };

    this.div = window.document.createElement("div");
    this.div.id = region.id; // useless?
    this.applyStyles(regionNodeStyles);
  }
  RegionNodeBox.prototype = _objCreate(StyleBox.prototype);
  RegionNodeBox.prototype.constructor = RegionNodeBox;

  function RegionCueStyleBox(window, cue) {
    StyleBox.call(this);
    this.cueDiv = parseContent(window, cue.text, PARSE_CONTENT_MODE.REGION_CUE);

    var regionCueStyles = {
      position: "relative",
      writingMode: "horizontal-tb",
      unicodeBidi: "plaintext",
      width: "auto",
      height: "auto",
      textAlign: cue.align,
    };
    // TODO: fix me, LTR and RTL ? using margin replace the "left/right"
    // 6.1.14.3.3
    var offset = cue.computedPosition * cue.region.width / 100;
    // 6.1.14.3.4
    switch (cue.align) {
      case "start":
      case "left":
        regionCueStyles.left = offset + "%";
        regionCueStyles.right = "auto";
        break;
      case "end":
      case "right":
        regionCueStyles.left = "auto";
        regionCueStyles.right = offset + "%";
        break;
      case "middle":
        break;
    }

    this.div = window.document.createElement("div");
    this.applyStyles(regionCueStyles);
    this.div.appendChild(this.cueDiv);
  }
  RegionCueStyleBox.prototype = _objCreate(StyleBox.prototype);
  RegionCueStyleBox.prototype.constructor = RegionCueStyleBox;

  // Represents the co-ordinates of an Element in a way that we can easily
  // compute things with such as if it overlaps or intersects with another Element.
  // Can initialize it with either a StyleBox or another BoxPosition.
  function BoxPosition(obj) {
    var isIE8 = (typeof navigator !== "undefined") &&
      (/MSIE\s8\.0/).test(navigator.userAgent);

    // Either a BoxPosition was passed in and we need to copy it, or a StyleBox
    // was passed in and we need to copy the results of 'getBoundingClientRect'
    // as the object returned is readonly. All co-ordinate values are in reference
    // to the viewport origin (top left).
    var lh, height, width, top;
    if (obj.div) {
      height = obj.div.offsetHeight;
      width = obj.div.offsetWidth;
      top = obj.div.offsetTop;

      var rects = (rects = obj.div.childNodes) && (rects = rects[0]) &&
                  rects.getClientRects && rects.getClientRects();
      obj = obj.div.getBoundingClientRect();
      // In certain cases the outter div will be slightly larger then the sum of
      // the inner div's lines. This could be due to bold text, etc, on some platforms.
      // In this case we should get the average line height and use that. This will
      // result in the desired behaviour.
      lh = rects ? Math.max((rects[0] && rects[0].height) || 0, obj.height / rects.length)
                 : 0;

    }
    this.left = obj.left;
    this.right = obj.right;
    this.top = obj.top || top;
    this.height = obj.height || height;
    this.bottom = obj.bottom || (top + (obj.height || height));
    this.width = obj.width || width;
    this.lineHeight = lh !== undefined ? lh : obj.lineHeight;

    if (isIE8 && !this.lineHeight) {
      this.lineHeight = 13;
    }
  }

  // Move the box along a particular axis. Optionally pass in an amount to move
  // the box. If no amount is passed then the default is the line height of the
  // box.
  BoxPosition.prototype.move = function(axis, toMove) {
    toMove = toMove !== undefined ? toMove : this.lineHeight;
    switch (axis) {
    case "+x":
      this.left += toMove;
      this.right += toMove;
      break;
    case "-x":
      this.left -= toMove;
      this.right -= toMove;
      break;
    case "+y":
      this.top += toMove;
      this.bottom += toMove;
      break;
    case "-y":
      this.top -= toMove;
      this.bottom -= toMove;
      break;
    }
  };

  // Check if this box overlaps another box, b2.
  BoxPosition.prototype.overlaps = function(b2) {
    return this.left < b2.right &&
           this.right > b2.left &&
           this.top < b2.bottom &&
           this.bottom > b2.top;
  };

  // Check if this box overlaps any other boxes in boxes.
  BoxPosition.prototype.overlapsAny = function(boxes) {
    for (var i = 0; i < boxes.length; i++) {
      if (this.overlaps(boxes[i])) {
        return true;
      }
    }
    return false;
  };

  // Check if this box is within another box.
  BoxPosition.prototype.within = function(container) {
    return this.top >= container.top &&
           this.bottom <= container.bottom &&
           this.left >= container.left &&
           this.right <= container.right;
  };

  // Check if this box is entirely within the container or it is overlapping
  // on the edge opposite of the axis direction passed. For example, if "+x" is
  // passed and the box is overlapping on the left edge of the container, then
  // return true.
  BoxPosition.prototype.overlapsOppositeAxis = function(container, axis) {
    switch (axis) {
    case "+x":
      return this.left < container.left;
    case "-x":
      return this.right > container.right;
    case "+y":
      return this.top < container.top;
    case "-y":
      return this.bottom > container.bottom;
    }
  };

  // Find the percentage of the area that this box is overlapping with another
  // box.
  BoxPosition.prototype.intersectPercentage = function(b2) {
    var x = Math.max(0, Math.min(this.right, b2.right) - Math.max(this.left, b2.left)),
        y = Math.max(0, Math.min(this.bottom, b2.bottom) - Math.max(this.top, b2.top)),
        intersectArea = x * y;
    return intersectArea / (this.height * this.width);
  };

  // Convert the positions from this box to CSS compatible positions using
  // the reference container's positions. This has to be done because this
  // box's positions are in reference to the viewport origin, whereas, CSS
  // values are in referecne to their respective edges.
  BoxPosition.prototype.toCSSCompatValues = function(reference) {
    return {
      top: this.top - reference.top,
      bottom: reference.bottom - this.bottom,
      left: this.left - reference.left,
      right: reference.right - this.right,
      height: this.height,
      width: this.width
    };
  };

  // Get an object that represents the box's position without anything extra.
  // Can pass a StyleBox, HTMLElement, or another BoxPositon.
  BoxPosition.getSimpleBoxPosition = function(obj) {
    var height = obj.div ? obj.div.offsetHeight : obj.tagName ? obj.offsetHeight : 0;
    var width = obj.div ? obj.div.offsetWidth : obj.tagName ? obj.offsetWidth : 0;
    var top = obj.div ? obj.div.offsetTop : obj.tagName ? obj.offsetTop : 0;

    obj = obj.div ? obj.div.getBoundingClientRect() :
                  obj.tagName ? obj.getBoundingClientRect() : obj;
    var ret = {
      left: obj.left,
      right: obj.right,
      top: obj.top || top,
      height: obj.height || height,
      bottom: obj.bottom || (top + (obj.height || height)),
      width: obj.width || width
    };
    return ret;
  };

  // Move a StyleBox to its specified, or next best, position. The containerBox
  // is the box that contains the StyleBox, such as a div. boxPositions are
  // a list of other boxes that the styleBox can't overlap with.
  function moveBoxToLinePosition(window, styleBox, containerBox, boxPositions) {

    // Find the best position for a cue box, b, on the video. The axis parameter
    // is a list of axis, the order of which, it will move the box along. For example:
    // Passing ["+x", "-x"] will move the box first along the x axis in the positive
    // direction. If it doesn't find a good position for it there it will then move
    // it along the x axis in the negative direction.
    function findBestPosition(b, axis) {
      var bestPosition,
          specifiedPosition = new BoxPosition(b),
          percentage = 1; // Highest possible so the first thing we get is better.

      for (var i = 0; i < axis.length; i++) {
        while (b.overlapsOppositeAxis(containerBox, axis[i]) ||
               (b.within(containerBox) && b.overlapsAny(boxPositions))) {
          b.move(axis[i]);
        }
        // We found a spot where we aren't overlapping anything. This is our
        // best position.
        if (b.within(containerBox)) {
          return b;
        }
        var p = b.intersectPercentage(containerBox);
        // If we're outside the container box less then we were on our last try
        // then remember this position as the best position.
        if (percentage > p) {
          bestPosition = new BoxPosition(b);
          percentage = p;
        }
        // Reset the box position to the specified position.
        b = new BoxPosition(specifiedPosition);
      }
      return bestPosition || specifiedPosition;
    }

    var boxPosition = new BoxPosition(styleBox),
        cue = styleBox.cue,
        linePos = cue.computedLine,
        axis = [];

    // If we have a line number to align the cue to.
    if (cue.snapToLines) {
      var size;
      switch (cue.vertical) {
      case "":
        axis = [ "+y", "-y" ];
        size = "height";
        break;
      case "rl":
        axis = [ "+x", "-x" ];
        size = "width";
        break;
      case "lr":
        axis = [ "-x", "+x" ];
        size = "width";
        break;
      }

      var step = boxPosition.lineHeight,
          position = step * Math.round(linePos),
          maxPosition = containerBox[size] + step,
          initialAxis = axis[0];

      if (step == 0) {
        return;
      }

      // If the specified intial position is greater then the max position then
      // clamp the box to the amount of steps it would take for the box to
      // reach the max position.
      if (Math.abs(position) > maxPosition) {
        position = position < 0 ? -1 : 1;
        position *= Math.ceil(maxPosition / step) * step;
      }

      // If computed line position returns negative then line numbers are
      // relative to the bottom of the video instead of the top. Therefore, we
      // need to increase our initial position by the length or width of the
      // video, depending on the writing direction, and reverse our axis directions.
      if (linePos < 0) {
        position += cue.vertical === "" ? containerBox.height : containerBox.width;
        axis = axis.reverse();
      }

      // Move the box to the specified position. This may not be its best
      // position.
      boxPosition.move(initialAxis, position);

    } else {
      // If we have a percentage line value for the cue.
      var calculatedPercentage = (boxPosition.lineHeight / containerBox.height) * 100;

      switch (cue.lineAlign) {
      case "center":
        linePos -= (calculatedPercentage / 2);
        break;
      case "end":
        linePos -= calculatedPercentage;
        break;
      }

      // Apply initial line position to the cue box.
      switch (cue.vertical) {
      case "":
        styleBox.applyStyles({
          top: styleBox.formatStyle(linePos, "%")
        });
        break;
      case "rl":
        styleBox.applyStyles({
          left: styleBox.formatStyle(linePos, "%")
        });
        break;
      case "lr":
        styleBox.applyStyles({
          right: styleBox.formatStyle(linePos, "%")
        });
        break;
      }

      axis = [ "+y", "-x", "+x", "-y" ];

      // Get the box position again after we've applied the specified positioning
      // to it.
      boxPosition = new BoxPosition(styleBox);
    }

    var bestPosition = findBestPosition(boxPosition, axis);
    styleBox.move(bestPosition.toCSSCompatValues(containerBox));
  }

  function WebVTT() {
    // Nothing
  }

  // Helper to allow strings to be decoded instead of the default binary utf8 data.
  WebVTT.StringDecoder = function() {
    return {
      decode: function(data) {
        if (!data) {
          return "";
        }
        if (typeof data !== "string") {
          throw new Error("Error - expected string data.");
        }
        return decodeURIComponent(encodeURIComponent(data));
      }
    };
  };

  WebVTT.convertCueToDOMTree = function(window, cuetext) {
    if (!window) {
      return null;
    }
    return parseContent(window, cuetext, PARSE_CONTENT_MODE.DOCUMENT_FRAGMENT);
  };

  var FONT_SIZE_PERCENT = 0.05;
  var FONT_STYLE = "sans-serif";

  // Runs the processing model over the cues and regions passed to it.
  // @param overlay A block level element (usually a div) that the computed cues
  //                and regions will be placed into.
  // @param controls  A Control bar element. Cues' position will be
  //                 affected and repositioned according to it.
  WebVTT.processCues = function(window, cues, overlay, controls) {
    if (!window || !cues || !overlay) {
      return null;
    }

    var controlBar;
    var controlBarShown;
    if (controls) {
      controlBar = controls.ownerDocument.getAnonymousElementByAttribute(
        controls, "anonid", "controlBar");
      controlBarShown = controlBar ? !!controlBar.clientHeight : false;
    }

    // Determine if we need to compute the display states of the cues. This could
    // be the case if a cue's state has been changed since the last computation or
    // if it has not been computed yet.
    function shouldCompute(cues) {
      if (overlay.lastControlBarShownStatus != controlBarShown) {
        return true;
      }

      for (var i = 0; i < cues.length; i++) {
        if (cues[i].hasBeenReset || !cues[i].displayState) {
          return true;
        }
      }
      return false;
    }

    // We don't need to recompute the cues' display states. Just reuse them.
    if (!shouldCompute(cues)) {
      return;
    }
    overlay.lastControlBarShownStatus = controlBarShown;

    // Remove all previous children.
    while (overlay.firstChild) {
      overlay.firstChild.remove();
    }
    var rootOfCues = window.document.createElement("div");
    rootOfCues.style.position = "absolute";
    rootOfCues.style.left = "0";
    rootOfCues.style.right = "0";
    rootOfCues.style.top = "0";
    rootOfCues.style.bottom = "0";
    overlay.appendChild(rootOfCues);

    var boxPositions = [],
        containerBox = BoxPosition.getSimpleBoxPosition(rootOfCues),
        fontSize = Math.round(containerBox.height * FONT_SIZE_PERCENT * 100) / 100;
    var styleOptions = {
      font: fontSize + "px " + FONT_STYLE
    };

    (function() {
      var styleBox, cue, controlBarBox;

      if (controlBarShown) {
        controlBarBox = BoxPosition.getSimpleBoxPosition(controlBar);
        // Add an empty output box that cover the same region as video control bar.
        boxPositions.push(controlBarBox);
      }

      // https://w3c.github.io/webvtt/#processing-model 6.1.12.1
      // Create regionNode
      var regionNodeBoxes = {};
      var regionNodeBox;

      for (var i = 0; i < cues.length; i++) {
        cue = cues[i];
        if (cue.region != null) {
         // 6.1.14.1
          styleBox = new RegionCueStyleBox(window, cue);

          if (!regionNodeBoxes[cue.region.id]) {
            // create regionNode
            // Adjust the container hieght to exclude the controlBar
            var adjustContainerBox = BoxPosition.getSimpleBoxPosition(rootOfCues);
            if (controlBarShown) {
              adjustContainerBox.height -= controlBarBox.height;
              adjustContainerBox.bottom += controlBarBox.height;
            }
            regionNodeBox = new RegionNodeBox(window, cue.region, adjustContainerBox);
            regionNodeBoxes[cue.region.id] = regionNodeBox;
          }
          // 6.1.14.3
          var currentRegionBox = regionNodeBoxes[cue.region.id];
          var currentRegionNodeDiv = currentRegionBox.div;
          // 6.1.14.3.2
          // TODO: fix me, it looks like the we need to set/change "top" attribute at the styleBox.div
          // to do the "scroll up", however, we do not implement it yet?
          if (cue.region.scroll == "up" && currentRegionNodeDiv.childElementCount > 0) {
            styleBox.div.style.transitionProperty = "top";
            styleBox.div.style.transitionDuration = "0.433s";
          }

          currentRegionNodeDiv.appendChild(styleBox.div);
          rootOfCues.appendChild(currentRegionNodeDiv);
          cue.displayState = styleBox.div;
          boxPositions.push(BoxPosition.getSimpleBoxPosition(currentRegionBox));
        } else {
          // Compute the intial position and styles of the cue div.
          styleBox = new CueStyleBox(window, cue, styleOptions);
          styleBox.cueDiv.style.setProperty("--cue-font-size", fontSize + "px");
          rootOfCues.appendChild(styleBox.div);

          // Move the cue div to it's correct line position.
          moveBoxToLinePosition(window, styleBox, containerBox, boxPositions);

          // Remember the computed div so that we don't have to recompute it later
          // if we don't have too.
          cue.displayState = styleBox.div;

          boxPositions.push(BoxPosition.getSimpleBoxPosition(styleBox));
        }
      }
    })();
  };

  WebVTT.Parser = function(window, decoder) {
    this.window = window;
    this.state = "INITIAL";
    this.buffer = "";
    this.decoder = decoder || new TextDecoder("utf8");
    this.regionList = [];
  };

  WebVTT.Parser.prototype = {
    // If the error is a ParsingError then report it to the consumer if
    // possible. If it's not a ParsingError then throw it like normal.
    reportOrThrowError: function(e) {
      if (e instanceof ParsingError) {
        this.onparsingerror && this.onparsingerror(e);
      } else {
        throw e;
      }
    },
    parse: function (data) {
      var self = this;

      // If there is no data then we won't decode it, but will just try to parse
      // whatever is in buffer already. This may occur in circumstances, for
      // example when flush() is called.
      if (data) {
        // Try to decode the data that we received.
        self.buffer += self.decoder.decode(data, {stream: true});
      }

      function collectNextLine() {
        var buffer = self.buffer;
        var pos = 0;
        while (pos < buffer.length && buffer[pos] !== '\r' && buffer[pos] !== '\n') {
          ++pos;
        }
        var line = buffer.substr(0, pos);
        // Advance the buffer early in case we fail below.
        if (buffer[pos] === '\r') {
          ++pos;
        }
        if (buffer[pos] === '\n') {
          ++pos;
        }
        self.buffer = buffer.substr(pos);
        // Spec defined replacement.
        return line.replace(/[\u0000]/g, "\uFFFD");
      }

      function createCueIfNeeded() {
        if (!self.cue) {
          self.cue = new self.window.VTTCue(0, 0, "");
        }
      }

      // Parsing cue identifier and the identifier should be unique.
      // Return true if the input is a cue identifier.
      function parseCueIdentifier(input) {
        if (maybeIsTimeStampFormat(line)) {
          self.state = "CUE";
          return false;
        }

        createCueIfNeeded();
        // TODO : ensure the cue identifier is unique among all cue identifiers.
        self.cue.id = containsTimeDirectionSymbol(input) ? "" : input;
        self.state = "CUE";
        return true;
      }

      // Parsing the timestamp and cue settings.
      // See spec, https://w3c.github.io/webvtt/#collect-webvtt-cue-timings-and-settings
      function parseCueMayThrow(input) {
        try {
          createCueIfNeeded();
          parseCue(input, self.cue, self.regionList);
          self.state = "CUETEXT";
        } catch (e) {
          self.reportOrThrowError(e);
          // In case of an error ignore rest of the cue.
          self.cue = null;
          self.state = "BADCUE";
        }
      }

      // 3.4 WebVTT region and WebVTT region settings syntax
      function parseRegion(input) {
        var settings = new Settings();

        parseOptions(input, function (k, v) {
          switch (k) {
          case "id":
            settings.set(k, v);
            break;
          case "width":
            settings.percent(k, v);
            break;
          case "lines":
            settings.digitsValue(k, v);
            break;
          case "regionanchor":
          case "viewportanchor":
            var xy = v.split(',');
            if (xy.length !== 2) {
              break;
            }
            // We have to make sure both x and y parse, so use a temporary
            // settings object here.
            var anchor = new Settings();
            anchor.percent("x", xy[0]);
            anchor.percent("y", xy[1]);
            if (!anchor.has("x") || !anchor.has("y")) {
              break;
            }
            settings.set(k + "X", anchor.get("x"));
            settings.set(k + "Y", anchor.get("y"));
            break;
          case "scroll":
            settings.alt(k, v, ["up"]);
            break;
          }
        }, /=/, /\s/);

        // Create the region, using default values for any values that were not
        // specified.
        if (settings.has("id")) {
          try {
            var region = new self.window.VTTRegion();
            region.id = settings.get("id", "");
            region.width = settings.get("width", 100);
            region.lines = settings.get("lines", 3);
            region.regionAnchorX = settings.get("regionanchorX", 0);
            region.regionAnchorY = settings.get("regionanchorY", 100);
            region.viewportAnchorX = settings.get("viewportanchorX", 0);
            region.viewportAnchorY = settings.get("viewportanchorY", 100);
            region.scroll = settings.get("scroll", "");
            // Register the region.
            self.onregion && self.onregion(region);
            // Remember the VTTRegion for later in case we parse any VTTCues that
            // reference it.
            self.regionList.push({
              id: settings.get("id"),
              region: region
            });
          } catch(e) {
            dump("VTTRegion Error " + e + "\n");
            var regionPref = Services.prefs.getBoolPref("media.webvtt.regions.enabled");
            dump("regionPref " + regionPref + "\n");
          }
        }
      }

      // Parsing the WebVTT signature, it contains parsing algo step1 to step9.
      // See spec, https://w3c.github.io/webvtt/#file-parsing
      function parseSignatureMayThrow(input) {
        let signature = collectNextLine();
        if (!/^WEBVTT([ \t].*)?$/.test(signature)) {
          throw new ParsingError(ParsingError.Errors.BadSignature);
        } else {
          self.state = "HEADER";
        }
      }

      // Parsing the region and style information.
      // See spec, https://w3c.github.io/webvtt/#collect-a-webvtt-block
      //
      // There are sereval things would appear in header,
      //   1. Region or Style setting
      //   2. Garbage (meaningless string)
      //   3. Empty line
      //   4. Cue's timestamp
      // The case 4 happens when there is no line interval between the header
      // and the cue blocks. In this case, we should preserve the line and
      // return it for the next phase parsing.
      function parseHeader() {
        let line = null;
        while (self.buffer && self.state === "HEADER") {
          line = collectNextLine();

          if (/^REGION|^STYLE/i.test(line)) {
            parseOptions(line, function (k, v) {
              switch (k.toUpperCase()) {
              case "REGION":
                parseRegion(v);
                break;
              case "STYLE":
                // TODO : not supported yet.
                break;
              }
            }, ":");
          } else if (maybeIsTimeStampFormat(line)) {
            self.state = "CUE";
            break;
          } else if (!line ||
                     onlyContainsWhiteSpaces(line) ||
                     containsTimeDirectionSymbol(line)) {
            // empty line, whitespaces or string contains "-->"
            break;
          }
        }

        // End parsing header part and doesn't see the timestamp.
        if (self.state === "HEADER") {
          self.state = "ID";
          line = null
        }
        return line;
      }

      // 5.1 WebVTT file parsing.
      try {
        if (self.state === "INITIAL") {
          parseSignatureMayThrow();
        }

        var line;
        if (self.state === "HEADER") {
          line = parseHeader();
        }

        var nextIteration = false;
        while (nextIteration || self.buffer) {
          nextIteration = false;
          if (!line) {
            // Since the data receiving is async, we need to wait until the
            // buffer gets the full line.
            if (!/\r\n|\n|\r/.test(self.buffer)) {
              return this;
            }
            line = collectNextLine();
          }

          switch (self.state) {
          case "ID":
            // Ignore NOTE and line terminator
            if (/^NOTE($|[ \t])/.test(line) || !line) {
              break;
            }
            // If there is no cue identifier, keep the line and reuse this line
            // in next iteration.
            if (!parseCueIdentifier(line)) {
              nextIteration = true;
              continue;
            }
            break;
          case "CUE":
            parseCueMayThrow(line);
            break;
          case "CUETEXT":
            // Report the cue when (1) get an empty line (2) get the "-->""
            if (!line || containsTimeDirectionSymbol(line)) {
              // We are done parsing self cue.
              self.oncue && self.oncue(self.cue);
              self.cue = null;
              self.state = "ID";
              // Keep the line and reuse this line in next iteration.
              nextIteration = true;
              continue;
            }
            if (self.cue.text) {
              self.cue.text += "\n";
            }
            self.cue.text += line;
            break;
          case "BADCUE": // BADCUE
            // 54-62 - Collect and discard the remaining cue.
            if (!line) {
              self.state = "ID";
            }
            break;
          }
          // The line was already parsed, empty it to ensure we can get the
          // new line in next iteration.
          line = null;
        }
      } catch (e) {
        self.reportOrThrowError(e);

        // If we are currently parsing a cue, report what we have.
        if (self.state === "CUETEXT" && self.cue && self.oncue) {
          self.oncue(self.cue);
        }
        self.cue = null;
        // Enter BADWEBVTT state if header was not parsed correctly otherwise
        // another exception occurred so enter BADCUE state.
        self.state = self.state === "INITIAL" ? "BADWEBVTT" : "BADCUE";
      }
      return this;
    },
    flush: function () {
      var self = this;
      try {
        // Finish decoding the stream.
        self.buffer += self.decoder.decode();
        // Synthesize the end of the current cue or region.
        if (self.cue || self.state === "HEADER") {
          self.buffer += "\n\n";
          self.parse();
        }
        // If we've flushed, parsed, and we're still on the INITIAL state then
        // that means we don't have enough of the stream to parse the first
        // line.
        if (self.state === "INITIAL") {
          throw new ParsingError(ParsingError.Errors.BadSignature);
        }
      } catch(e) {
        self.reportOrThrowError(e);
      }
      self.onflush && self.onflush();
      return this;
    }
  };

  global.WebVTT = WebVTT;

}(this));
