# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# This script generates jit/LOpcodes.h (list of LIR instructions) and
# jit/MOpcodes.h (list of MIR instructions) from MIR.h and LIR files.

import re

HEADER_TEMPLATE = """\
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef %(includeguard)s
#define %(includeguard)s

/* This file is generated by jit/GenerateOpcodeFiles.py. Do not edit! */

#define %(listname)s(_) \\
%(ops)s

#endif // %(includeguard)s
"""

def get_opcodes(inputs, pat):
  # Preserve the original order. Use a set to detect duplicates.
  ops = []
  ops_set = set()
  for inputfile in inputs:
    for line in open(inputfile):
      match = pat.match(line)
      if match:
        op = match.group('name')
        if op in ops_set:
          raise Exception("Duplicate opcode {} in {}".format(op, inputfile))
        ops.append(op)
        ops_set.add(op)
  assert len(ops) == len(ops_set)
  return ops

def generate_header(c_out, inputs, pat, includeguard, listname):
  ops = get_opcodes(inputs, pat)
  ops_string = '\\\n'.join(['_(' + op + ')' for op in ops])
  c_out.write(HEADER_TEMPLATE % {
    'ops': ops_string,
    'includeguard': includeguard,
    'listname': listname,
  })

def generate_mir_header(c_out, *inputs):
  pat = re.compile(r"^\s*INSTRUCTION_HEADER(_WITHOUT_TYPEPOLICY)?\((?P<name>\w+)\);?$")
  generate_header(c_out, inputs, pat, 'jit_MOpcodes_h', 'MIR_OPCODE_LIST')

def generate_lir_header(c_out, *inputs):
  pat = re.compile(r"^\s*LIR_HEADER\((?P<name>\w+)\);?$")
  generate_header(c_out, inputs, pat, 'jit_LOpcodes_h', 'LIR_OPCODE_LIST')
