/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */
"use strict";

var ConsoleAPI = {
  observe: function observe(aMessage, aTopic, aData) {
    aMessage = aMessage.wrappedJSObject;

    let mappedArguments = Array.from(
      aMessage.arguments,
      this.formatResult,
      this
    );
    let joinedArguments = mappedArguments.join(" ");

    if (aMessage.level == "error" || aMessage.level == "warn") {
      let flag =
        aMessage.level == "error"
          ? Ci.nsIScriptError.errorFlag
          : Ci.nsIScriptError.warningFlag;
      let consoleMsg = Cc["@mozilla.org/scripterror;1"].createInstance(
        Ci.nsIScriptError
      );
      consoleMsg.init(
        joinedArguments,
        null,
        null,
        0,
        0,
        flag,
        "content javascript"
      );
      Services.console.logMessage(consoleMsg);
    } else if (aMessage.level == "trace") {
      let bundle = Services.strings.createBundle(
        "chrome://browser/locale/browser.properties"
      );
      let args = aMessage.arguments;
      let filename = this.abbreviateSourceURL(args[0].filename);
      let functionName =
        args[0].functionName ||
        bundle.GetStringFromName("stacktrace.anonymousFunction");
      let lineNumber = args[0].lineNumber;

      let body = bundle.formatStringFromName("stacktrace.outputMessage", [
        filename,
        functionName,
        lineNumber,
      ]);
      body += "\n";
      args.forEach(function(aFrame) {
        let functionName =
          aFrame.functionName ||
          bundle.GetStringFromName("stacktrace.anonymousFunction");
        body +=
          "  " +
          aFrame.filename +
          " :: " +
          functionName +
          " :: " +
          aFrame.lineNumber +
          "\n";
      });

      Services.console.logStringMessage(body);
    } else if (aMessage.level == "time" && aMessage.arguments) {
      let bundle = Services.strings.createBundle(
        "chrome://browser/locale/browser.properties"
      );
      let body = bundle.formatStringFromName("timer.start", [
        aMessage.arguments.name,
      ]);
      Services.console.logStringMessage(body);
    } else if (aMessage.level == "timeEnd" && aMessage.arguments) {
      let bundle = Services.strings.createBundle(
        "chrome://browser/locale/browser.properties"
      );
      let body = bundle.formatStringFromName("timer.end", [
        aMessage.arguments.name,
        aMessage.arguments.duration,
      ]);
      Services.console.logStringMessage(body);
    } else if (
      ["group", "groupCollapsed", "groupEnd"].includes(aMessage.level)
    ) {
      // Do nothing yet
    } else {
      Services.console.logStringMessage(joinedArguments);
    }
  },

  getResultType: function getResultType(aResult) {
    let type = aResult === null ? "null" : typeof aResult;
    if (type == "object" && aResult.constructor && aResult.constructor.name) {
      type = aResult.constructor.name;
    }
    return type.toLowerCase();
  },

  formatResult: function formatResult(aResult) {
    let output = "";
    let type = this.getResultType(aResult);
    switch (type) {
      case "string":
      case "boolean":
      case "date":
      case "error":
      case "number":
      case "regexp":
        output = aResult.toString();
        break;
      case "null":
      case "undefined":
        output = type;
        break;
      default:
        output = aResult.toString();
        break;
    }

    return output;
  },

  abbreviateSourceURL: function abbreviateSourceURL(aSourceURL) {
    // Remove any query parameters.
    let hookIndex = aSourceURL.indexOf("?");
    if (hookIndex > -1) {
      aSourceURL = aSourceURL.substring(0, hookIndex);
    }

    // Remove a trailing "/".
    if (aSourceURL[aSourceURL.length - 1] == "/") {
      aSourceURL = aSourceURL.substring(0, aSourceURL.length - 1);
    }

    // Remove all but the last path component.
    let slashIndex = aSourceURL.lastIndexOf("/");
    if (slashIndex > -1) {
      aSourceURL = aSourceURL.substring(slashIndex + 1);
    }

    return aSourceURL;
  },
};
