/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */
"use strict";

var OfflineApps = {
  offlineAppRequested: function(aContentWindow) {
    if (!Services.prefs.getBoolPref("browser.offline-apps.notify")) {
      return;
    }

    let tab = BrowserApp.getTabForWindow(aContentWindow);
    let principal = aContentWindow.document.nodePrincipal;
    let currentURI = aContentWindow.document.documentURIObject;

    // Don't bother showing UI if the user has already made a decision
    if (
      Services.perms.testExactPermissionFromPrincipal(
        principal,
        "offline-app"
      ) != Services.perms.UNKNOWN_ACTION
    ) {
      return;
    }

    try {
      if (Services.prefs.getBoolPref("offline-apps.allow_by_default")) {
        // All pages can use offline capabilities, no need to ask the user
        return;
      }
    } catch (e) {
      // This pref isn't set by default, ignore failures
    }

    let host = currentURI.asciiHost;
    let notificationID = "offline-app-requested-" + host;

    let strings = Strings.browser;
    let buttons = [
      {
        label: strings.GetStringFromName("offlineApps.dontAllow2"),
        callback: function(aChecked) {
          if (aChecked) {
            OfflineApps.disallowSite(aContentWindow.document);
          }
        },
      },
      {
        label: strings.GetStringFromName("offlineApps.allow"),
        callback: function() {
          OfflineApps.allowSite(aContentWindow.document);
        },
        positive: true,
      },
    ];

    let requestor = BrowserApp.manifest
      ? "'" + BrowserApp.manifest.name + "'"
      : host;
    let message = strings.formatStringFromName("offlineApps.ask", [requestor]);
    let options = {
      checkbox: Strings.browser.GetStringFromName("offlineApps.dontAskAgain"),
    };
    NativeWindow.doorhanger.show(
      message,
      notificationID,
      buttons,
      tab.id,
      options
    );
  },

  allowSite: function(aDocument) {
    Services.perms.addFromPrincipal(
      aDocument.nodePrincipal,
      "offline-app",
      Services.perms.ALLOW_ACTION
    );

    // When a site is enabled while loading, manifest resources will
    // start fetching immediately.  This one time we need to do it
    // ourselves.
    this._startFetching(aDocument);
  },

  disallowSite: function(aDocument) {
    Services.perms.addFromPrincipal(
      aDocument.nodePrincipal,
      "offline-app",
      Services.perms.DENY_ACTION
    );
  },

  _startFetching: function(aDocument) {
    if (!aDocument.documentElement) {
      return;
    }

    let manifest = aDocument.documentElement.getAttribute("manifest");
    if (!manifest) {
      return;
    }

    let manifestURI = Services.io.newURI(
      manifest,
      aDocument.characterSet,
      aDocument.documentURIObject
    );
    let updateService = Cc[
      "@mozilla.org/offlinecacheupdate-service;1"
    ].getService(Ci.nsIOfflineCacheUpdateService);
    updateService.scheduleUpdate(
      manifestURI,
      aDocument.documentURIObject,
      aDocument.nodePrincipal,
      window
    );
  },
};
