/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @flow

import type {
  SourcePacket,
  PausedPacket,
  ThreadFront,
  Target,
  DebuggerClient,
} from "./types";

import Actions from "../../actions";

import { createPause, prepareSourcePayload } from "./create";
import sourceQueue from "../../utils/source-queue";
import { recordEvent } from "../../utils/telemetry";
import { prefs, features } from "../../utils/prefs";

const {
  WorkersListener,
  // $FlowIgnore
} = require("devtools/client/shared/workers-listener.js");

type Dependencies = {
  threadFront: ThreadFront,
  tabTarget: Target,
  actions: typeof Actions,
  debuggerClient: DebuggerClient,
};

let actions: typeof Actions;
let isInterrupted: boolean;

function addThreadEventListeners(thread: ThreadFront) {
  Object.keys(clientEvents).forEach(eventName => {
    thread.on(eventName, clientEvents[eventName].bind(null, thread));
  });
}

function attachAllTargets(currentTarget: Target) {
  return prefs.fission && currentTarget.chrome && !currentTarget.isAddon;
}

function setupEvents(dependencies: Dependencies) {
  const { tabTarget, threadFront, debuggerClient } = dependencies;
  actions = dependencies.actions;
  sourceQueue.initialize(actions);

  addThreadEventListeners(threadFront);
  tabTarget.on("workerListChanged", () => threadListChanged("worker"));
  debuggerClient.mainRoot.on("processListChanged", () =>
    threadListChanged("contentProcess")
  );

  if (features.windowlessServiceWorkers || attachAllTargets(tabTarget)) {
    const workersListener = new WorkersListener(debuggerClient.mainRoot);
    workersListener.addListener(() => threadListChanged("worker"));
  }
}

async function paused(threadFront: ThreadFront, packet: PausedPacket) {
  // When reloading we might get pauses from threads before they have been
  // added to the store. Ensure the pause won't be processed until we've
  // finished adding the thread.
  await actions.ensureHasThread(threadFront.actor);

  // If paused by an explicit interrupt, which are generated by the
  // slow script dialog and internal events such as setting
  // breakpoints, ignore the event.
  const { why } = packet;
  if (why.type === "interrupted" && !packet.why.onNext) {
    isInterrupted = true;
    return;
  }

  if (why.type == "alreadyPaused") {
    return;
  }

  if (packet.frame) {
    // When reloading we might receive a pause event before the
    // top frame's source has arrived.
    await actions.ensureSourceActor(
      threadFront.actorID,
      packet.frame.where.actor
    );
  }

  const pause = createPause(threadFront.actor, packet);

  actions.paused(pause);
  recordEvent("pause", { reason: why.type });
}

function resumed(threadFront: ThreadFront) {
  // NOTE: the client suppresses resumed events while interrupted
  // to prevent unintentional behavior.
  // see [client docs](../README.md#interrupted) for more information.
  if (isInterrupted) {
    isInterrupted = false;
    return;
  }

  actions.resumed(threadFront.actorID);
}

function newSource(threadFront: ThreadFront, { source }: SourcePacket) {
  sourceQueue.queue({
    type: "generated",
    data: prepareSourcePayload(threadFront, source),
  });
}

function threadListChanged(type) {
  actions.updateThreads(type);
}

function replayFramePositions(
  threadFront: ThreadFront,
  { positions, frame, thread }: Object
) {
  actions.setFramePositions(positions, frame, thread);
}

const clientEvents = {
  paused,
  resumed,
  newSource,
  replayFramePositions,
};

export { setupEvents, clientEvents, addThreadEventListeners, attachAllTargets };
