/* Any copyright is dedicated to the Public Domain.
 * https://creativecommons.org/publicdomain/zero/1.0/ */

use super::IMPLAUSIBILITY_PENALTY;
use encoding_rs::Encoding;
use encoding_rs::IBM866_INIT;
use encoding_rs::ISO_8859_13_INIT;
use encoding_rs::ISO_8859_2_INIT;
use encoding_rs::ISO_8859_4_INIT;
use encoding_rs::ISO_8859_5_INIT;
use encoding_rs::ISO_8859_6_INIT;
use encoding_rs::ISO_8859_7_INIT;
use encoding_rs::ISO_8859_8_INIT;
use encoding_rs::KOI8_U_INIT;
use encoding_rs::WINDOWS_1250_INIT;
use encoding_rs::WINDOWS_1251_INIT;
use encoding_rs::WINDOWS_1252_INIT;
use encoding_rs::WINDOWS_1253_INIT;
use encoding_rs::WINDOWS_1254_INIT;
use encoding_rs::WINDOWS_1255_INIT;
use encoding_rs::WINDOWS_1256_INIT;
use encoding_rs::WINDOWS_1257_INIT;
use encoding_rs::WINDOWS_1258_INIT;
use encoding_rs::WINDOWS_874_INIT;

#[repr(align(64))] // Align to cache lines
pub struct DetectorData {
    pub frequent_simplified: [u16; 128],
    pub frequent_kanji: [u16; 128],
    pub frequent_hangul: [u16; 128],
    latin_ascii: [u8; 128],
    non_latin_ascii: [u8; 128],
    turkish_ascii: [u8; 128],
    hebrew_ascii: [u8; 128],
    windows_1258: [u8; 128],
    windows_1250: [u8; 128],
    iso_8859_2: [u8; 128],
    windows_1251: [u8; 128],
    koi8_u: [u8; 128],
    iso_8859_5: [u8; 128],
    ibm866: [u8; 128],
    windows_1252: [u8; 128],
    windows_1252_icelandic: [u8; 128],
    windows_1253: [u8; 128],
    iso_8859_7: [u8; 128],
    windows_1254: [u8; 128],
    windows_1255: [u8; 128],
    iso_8859_8: [u8; 128],
    windows_1256: [u8; 128],
    iso_8859_6: [u8; 128],
    windows_1257: [u8; 128],
    iso_8859_13: [u8; 128],
    iso_8859_4: [u8; 128],
    windows_874: [u8; 128],
    vietnamese: [u8; 2697],
    central: [u8; 4545],
    cyrillic: [u8; 3016],
    western: [u8; 3705],
    icelandic: [u8; 1332],
    greek: [u8; 1840],
    turkish: [u8; 1387],
    hebrew: [u8; 3465],
    arabic: [u8; 4615],
    baltic: [u8; 2580],
    thai: [u8; 6075],
}

#[rustfmt::skip]
pub static DETECTOR_DATA: DetectorData = DetectorData {
    frequent_simplified: [
        0x7684, 0x5E74, 0x56FD, 0x65E5, 0x6708, 0x4E2D, 0x4EBA, 0x4E00, 0x5927, 0x4E3A, 0x5728, 0x662F, 0x5B66, 0x6709, 0x884C, 0x4F1A, 
        0x65AF, 0x4E8E, 0x5730, 0x533A, 0x6587, 0x548C, 0x5C14, 0x540D, 0x7B2C, 0x516C, 0x65F6, 0x5C0F, 0x90E8, 0x4E0D, 0x5E02, 0x53F0, 
        0x4EE5, 0x4E0A, 0x540E, 0x52A8, 0x51FA, 0x4E2A, 0x672C, 0x4F5C, 0x5BB6, 0x65B0, 0x6210, 0x897F, 0x5B9A, 0x91CD, 0x751F, 0x4E4B, 
        0x7535, 0x4E3B, 0x5B50, 0x7528, 0x7279, 0x5206, 0x6C11, 0x4E86, 0x4E9A, 0x5458, 0x514B, 0x5357, 0x653F, 0x7AD9, 0x5FB7, 0x4E0E, 
        0x7403, 0x4E1C, 0x79D1, 0x91CC, 0x9053, 0x5C71, 0x6CD5, 0x65B9, 0x5317, 0x5411, 0x5929, 0x53D1, 0x7269, 0x6765, 0x5230, 0x673A, 
        0x661F, 0x8DEF, 0x76EE, 0x7F8E, 0x6751, 0x9AD8, 0x957F, 0x519B, 0x5229, 0x4E09, 0x62C9, 0x8F66, 0x5DDE, 0x57FA, 0x6D77, 0x81EA, 
        0x4E0B, 0x8D5B, 0x9762, 0x52A0, 0x4ED6, 0x9A6C, 0x5176, 0x53C2, 0x53BF, 0x4EE3, 0x5185, 0x7406, 0x4E16, 0x4E8C, 0x7EBF, 0x53CA, 
        0x5EFA, 0x8868, 0x4F4D, 0x7F57, 0x7531, 0x7ACB, 0x591A, 0x53EF, 0x534E, 0x6797, 0x7EF4, 0x5EA6, 0x4E8B, 0x5E73, 0x5916, 0x4F53, 
    ],
    frequent_kanji: [
        0x5E74, 0x65E5, 0x6708, 0x5927, 0x672C, 0x5B66, 0x4EBA, 0x56FD, 0x4F1A, 0x4E2D, 0x51FA, 0x4E00, 0x8005, 0x5E02, 0x4F5C, 0x540D, 
        0x90E8, 0x7528, 0x5730, 0x884C, 0x5834, 0x7530, 0x7B2C, 0x751F, 0x5408, 0x5B50, 0x9053, 0x4E0A, 0x6771, 0x6642, 0x770C, 0x4EE3, 
        0x5C71, 0x793E, 0x4E8B, 0x753B, 0x65B0, 0x624B, 0x9AD8, 0x6210, 0x6226, 0x7269, 0x5F8C, 0x767A, 0x9577, 0x7ACB, 0x5206, 0x5DDD, 
        0x8A18, 0x6821, 0x9593, 0x696D, 0x95A2, 0x6240, 0x5B9A, 0x9078, 0x5C0F, 0x76EE, 0x52D5, 0x548C, 0x6587, 0x91CE, 0x540C, 0x524D, 
        0x5185, 0x958B, 0x7DDA, 0x81EA, 0x53F7, 0x516C, 0x99C5, 0x9001, 0x56DE, 0x753A, 0x9664, 0x4E3B, 0x5BB6, 0x5229, 0x8ECA, 0x901A, 
        0x4EAC, 0x8868, 0x5CF6, 0x4E0B, 0x4E16, 0x65B9, 0x6751, 0x66F8, 0x5168, 0x660E, 0x9023, 0x5E73, 0x653E, 0x4F53, 0x7684, 0x5F0F, 
        0x756A, 0x5EA6, 0x5317, 0x5165, 0x5916, 0x983C, 0x8A9E, 0x5973, 0x8A71, 0x6A5F, 0x8A2D, 0x539F, 0x4E09, 0x524A, 0x533A, 0x6D77, 
        0x4F9D, 0x5F53, 0x73FE, 0x5BFE, 0x4F4D, 0x6570, 0x5316, 0x795E, 0x66F2, 0x7406, 0x6559, 0x7279, 0x7248, 0x5728, 0x6CD5, 0x898B, 
    ],
    frequent_hangul: [
        0xC774, 0xC758, 0xB2E4, 0xAE30, 0xC5D0, 0xB85C, 0xC0AC, 0xB144, 0xC2A4, 0xB9AC, 0xB294, 0xC77C, 0xD558, 0xAC00, 0xC2DC, 0xC9C0, 
        0xB300, 0xC11C, 0xBD84, 0xAD6D, 0xD55C, 0xB3C4, 0xC778, 0xACE0, 0xB958, 0xC790, 0xC8FC, 0xC544, 0xC744, 0xB77C, 0xC218, 0xC81C, 
        0xC815, 0xC6D4, 0xB098, 0xAD6C, 0xC804, 0xC5B4, 0xC740, 0xADF8, 0xBD80, 0xB97C, 0xB3D9, 0xC120, 0xC73C, 0xBB38, 0xD2B8, 0xC6A9, 
        0xBCF4, 0xC704, 0xB4DC, 0xACFC, 0xAD50, 0xC0C1, 0xB9C8, 0xC7A5, 0xD559, 0xC6D0, 0xC131, 0xD654, 0xC5ED, 0xB2C8, 0xBBF8, 0xACF5, 
        0xACBD, 0xD574, 0xC624, 0xC6B0, 0xBA85, 0xC788, 0xD06C, 0xC601, 0xC18C, 0xC870, 0xD68C, 0xC5EC, 0xBBFC, 0xD1A0, 0xBE44, 0xC138, 
        0xB974, 0xC720, 0xC2E0, 0xD0A4, 0xC911, 0xACC4, 0xD0C0, 0xC5F0, 0xD504, 0xAD00, 0xB418, 0xC801, 0xCE58, 0xB808, 0xCE74, 0xC9C4, 
        0xC640, 0xD130, 0xB4E4, 0xBAA9, 0xACA8, 0xAC8C, 0xAC1C, 0xBC29, 0xD30C, 0xC0B0, 0xD638, 0xCD9C, 0xC74C, 0xB9BC, 0xBA74, 0xC791, 
        0xB9CC, 0xB2E8, 0xB118, 0xBAA8, 0xC694, 0xC5C8, 0xC0DD, 0xB0A8, 0xC7AC, 0xBB34, 0xD6C4, 0xD45C, 0xAD70, 0xD3EC, 0xB2F9, 0xB178, 
    ],
    latin_ascii: [
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,
          0,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,
        145,146,147,148,149,150,151,152,153,154,155,  0,  0,  0,  0,  0,
          0,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16,
         17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27,  0,  0,  0,  0,  0,
    ],
    non_latin_ascii: [
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,
          0,130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,
        130,130,130,130,130,130,130,130,130,130,130,  0,  0,  0,  0,  0,
          0,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
          2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  0,  0,  0,  0,  0,
    ],
    turkish_ascii: [
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,
          0,130,131,132,133,134,135,136,137,167,138,139,140,141,142,143,
        144,145,146,147,148,149,150,151,152,153,154,  0,  0,  0,  0,  0,
          0,  2,  3,  4,  5,  6,  7,  8,  9, 38, 10, 11, 12, 13, 14, 15,
         16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26,  0,  0,  0,  0,  0,
    ],
    hebrew_ascii: [
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
          0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  3,  0,
          1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  3,  3,  0,  0,  0,  3,
          0,130,130,130,130,130,130,130,130,130,130,130,130,130,130,130,
        130,130,130,130,130,130,130,130,130,130,130,  0,  0,  0,  0,  0,
          0,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
          2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  0,  0,  0,  0,  0,
    ],
    windows_1258: [
          0,255,  0, 55,  0,  0,  0,  0,  0,  0,255,  0,156,255,255,255,
        255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0, 28,255,255,181,
          0, 57, 55, 55, 55, 55, 54, 55, 55, 57, 55, 55, 58, 54, 56, 55,
         57, 58, 56, 56, 55, 57, 56, 54, 55, 56, 55, 55, 57, 57, 57, 57,
        161,162,163,164,156,156,156,160,165,166,167,156, 29,169,170,171,
        172,160, 30,174,175,176,156, 58,156,177,178,179,156,180, 31, 32,
         33, 34, 35, 36, 28, 28, 28, 32, 37, 38, 39, 28, 40, 41, 42, 43,
         44, 32, 45, 46, 47, 48, 28, 58, 28, 49, 50, 51, 28, 52,  0, 53,
    ],
    windows_1250: [
          0,255,  0,255,  0,  0,  0,  0,255,  0,156,  0,157,158,159,160,
        255,  0,  0,  0,  0,  0,  0,  0,255,  0, 28,  0, 29, 30, 31, 32,
          0, 70, 70,161, 70,162, 69, 70, 70, 72,163, 69, 70, 69, 71,165,
         72, 70, 70, 33, 70, 72, 71, 69, 70, 34, 35, 69,164,  0, 36, 37,
        167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,
        183,184,185,186,187,188,189, 70,190,191,192,193,194,195,196, 38,
         39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54,
         55, 56, 57, 58, 59, 60, 61, 70, 62, 63, 64, 65, 66, 67, 68,  0,
    ],
    iso_8859_2: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,162, 70,161, 70,164,157, 70, 70,156,163,158,160, 69,159,165,
         72, 34, 70, 33, 70, 36, 29, 70, 70, 28, 35, 30, 32,  0, 31, 37,
        167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,
        183,184,185,186,187,188,189, 70,190,191,192,193,194,195,196, 38,
         39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54,
         55, 56, 57, 58, 59, 60, 61, 70, 62, 63, 64, 65, 66, 67, 68,  0,
    ],
    windows_1251: [
        137,136,  0,  8,  0,  0,  0,  0,  0,  0,138,  0,139,140,141,142,
          9,  0,  0,  0,  0,  0,  0,  0,255,  0, 10,  0, 11, 12, 13, 14,
          0,143, 15,148,  4,145,  3,  4,146,  6,147,  6,  7,  3,  5,150,
          6,  7,144, 16, 17,  6,  5,  3, 18,  4, 19,  5, 20,149, 21, 22,
        151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,
        167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,
         23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
         39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54,
    ],
    koi8_u: [
          4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,
          4,  4,  4,  4,  4,  3,  0,  7,  7,  7,  0,  4,  6,  5,  3,  7,
          4,  4,  4, 18, 19,  4, 16, 22,  4,  4,  4,  4,  4, 17, 15,  4,
          4,  4,  4,146,147,  4,144,150,  4,  4,  4,  4,  4,145,143,  6,
         53, 23, 24, 45, 27, 28, 43, 26, 44, 31, 32, 33, 34, 35, 36, 37,
         38, 54, 39, 40, 41, 42, 29, 25, 51, 50, 30, 47, 52, 48, 46, 49,
        181,151,152,173,155,156,171,154,172,159,160,161,162,163,164,165,
        166,182,167,168,169,170,157,153,179,178,158,175,180,176,174,177,
    ],
    iso_8859_5: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,146,137,136,147,149,144,150,148,138,139,141,140,  3,143,142,
        151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,
        167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,
         23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
         39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54,
          4, 18,  9,  8, 19, 21, 16, 22, 20, 10, 11, 13, 12,  4, 15, 14,
    ],
    ibm866: [
        151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,
        167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,
         23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
          4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,
          4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,
          4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  4,
         39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54,
        146, 18,147, 19,150, 22,143, 15,  6,  3,  3,  0,  4,  4,  4,  0,
    ],
    windows_1252: [
          0,255,  0, 62,  0,  0,  0,  0,  0,  0,156,  0,157,255,187,255,
        255,  0,  0,  0,  0,  0,  0,  0,  0,  0, 28,  0, 29,255, 59,188,
          0, 64, 62, 62, 62, 62, 61, 62, 62, 64, 66, 61, 65, 61, 63, 62,
         64, 65, 63, 63, 62, 64, 63, 61, 62, 63, 66, 61, 64, 64, 64, 64,
        159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,
        186,175,176,177,178,179,180, 65,181,182,183,184,185,186,186, 30,
         31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46,
         58, 47, 48, 49, 50, 51, 52, 65, 53, 54, 55, 56, 57, 58, 58, 60,
    ],
    windows_1252_icelandic: [
          0,255,  0, 42,  0,  0,  0,  0,  0,  0,156,  0,156,255,156,255,
        255,  0,  0,  0,  0,  0,  0,  0,  0,  0, 28,  0, 28,255, 28,156,
          0, 44, 42, 42, 42, 42, 41, 42, 42, 44, 42, 42, 45, 41, 43, 42,
         44, 45, 43, 43, 42, 44, 43, 41, 42, 43, 42, 42, 44, 44, 44, 44,
        156,157,156,156,158,156,159,156,156,160,156,156,156,161,156,156,
        162,156,156,163,156,156,164, 45,165,156,166,156,156,167,168, 28,
         28, 29, 28, 28, 30, 28, 31, 28, 28, 32, 28, 28, 28, 33, 28, 28,
         34, 28, 28, 35, 28, 28, 36, 45, 37, 28, 38, 28, 28, 39, 40, 28,
    ],
    windows_1253: [
          4,255,  0,  4,  0,  0,  0,  0,255,  0,255,  0,255,255,255,255,
        255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,255,255,
          0,  4,136,  4,  4,  4,  3,  4,  4,  6,255,  6,  3,  3,  5,  3,
          6,  3,  5,  5,  0,  6,  5,  3,137,138,139,  5,168,  6,169,170,
          7,141,142,143,144,145,146,147,148,149,150,151,134,153,154,155,
        156,157,255,159,160,161,162,163,164,165,166,167,  8,  9, 10, 11,
         12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27,
         28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,255,
    ],
    iso_8859_7: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,  0,  0,  4,  4,  0,  3,  4,  4,  6,  0,  6,  3,  3,255,  3,
          6,  3,  5,  5,  0,  4,136,  3,137,138,139,  5,168,  6,169,170,
          7,141,142,143,144,145,146,147,148,149,150,151,134,153,154,155,
        156,157,255,159,160,161,162,163,164,165,166,167,  8,  9, 10, 11,
         12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27,
         28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,255,
    ],
    windows_1254: [
          0,255,  0, 42,  0,  0,  0,  0,  0,  0,155,  0,156,255,255,255,
        255,  0,  0,  0,  0,  0,  0,  0,  0,  0, 27,  0, 28,255,255,156,
          0, 44, 42, 42, 42, 42, 41, 42, 42, 44, 42, 44, 45, 41, 43, 42,
         44, 45, 43, 43, 42, 44, 43, 41, 42, 43, 42, 43, 44, 44, 44, 44,
        156,156,156,156,158,156,156,159,156,156,160,156,156,156,161,156,
        162,157,156,156,156,156,163, 45,156,156,156,164,165,166,168, 29,
         28, 28, 28, 28, 30, 28, 28, 31, 28, 28, 32, 28, 28, 28, 33, 28,
         34, 29, 28, 28, 28, 28, 35, 45, 28, 28, 28, 36, 37, 39, 40, 28,
    ],
    windows_1255: [
          0,255,  0,  5,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,
        255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,
          0,  7,  5,  5,  9,  5,  4,  5,  5,  7,  8,  8,  8,  4,  6,  5,
          7,  8,  6,  6,  5,  7,  6,  4,  5,  6,  8,  8,  7,  7,  7,  7,
         11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,  0,  0,
          0, 25, 26,  0, 27, 28, 29, 30, 31,255,255,255,255,255,255,255,
         32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
         48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58,255,255, 10, 10,255,
    ],
    iso_8859_8: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,255,  5,  5,  5,  5,  4,  5,  5,  7,  8,  8,  8,  4,  6,  5,
          7,  8,  6,  6,  5,  7,  6,  4,  5,  6,  8,  8,  7,  7,  7,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  4,
         32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
         48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58,255,255, 10, 10,255,
    ],
    windows_1256: [
          0, 11,  0,  7,  0,  0,  0,  0,  0,  0, 12,  0,130, 13, 14, 15,
         16,  0,  0,  0,  0,  0,  0,  0, 17,  0, 18,  0,  2,  3,  4, 19,
          0,  5,  7,  7,  7,  7,  6,  7,  7,  9, 20,  9, 10,  6,  8,  7,
          9, 10,  8,  8,  7,  9,  8,  6,  7,  8,  5,  8,  9,  9,  9,  5,
         21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
         37, 38, 39, 40, 41, 42, 43, 10, 44, 45, 46, 47, 48, 49, 50, 51,
          2, 52,  2, 53, 54, 55, 56,  2,  2,  2,  2,  2, 57, 58,  2,  2,
         59, 60, 61, 62,  2, 63, 64, 10, 65,  2, 66,  2,  2,  0,  0, 67,
    ],
    iso_8859_6: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,255,255,255,  7,255,255,255,255,255,255,255,  5,  6,255,255,
        255,255,255,255,255,255,255,255,255,255,255,  5,255,255,255,  5,
        255, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36,
         37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,255,255,255,255,255,
         48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
         64, 65, 66,255,255,255,255,255,255,255,255,255,255,255,255,255,
    ],
    windows_1257: [
          0,255,  0,255,  0,  0,  0,  0,255,  0,255,  0,255, 54, 54, 54,
        255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255, 54, 54,255,
          0,255, 54, 54, 54,255, 53, 54,156, 56,157, 56, 57, 53, 55,156,
         56, 57, 55, 55, 54, 56, 55, 53, 28, 55, 29, 55, 56, 56, 56, 28,
        159,160,161,158,162,156,163,164,165,166,158,167,168,169,170,171,
        172,158,173,156,174,175,176, 57,177,158,158,178,179,158,180, 30,
         31, 32, 33, 30, 34, 28, 35, 36, 37, 38, 30, 39, 40, 41, 42, 43,
         44, 30, 45, 28, 46, 47, 48, 57, 49, 30, 30, 50, 51, 30, 52,  0,
    ],
    iso_8859_13: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,  0, 54, 54, 54,  0, 53, 54,156, 56,157, 56, 57, 53, 55,156,
         56, 57, 55, 55,  0, 56, 55, 53, 28, 55, 29, 55, 56, 56, 56, 28,
        159,160,161,158,162,156,163,164,165,166,158,167,168,169,170,171,
        172,158,173,156,174,175,176, 57,177,158,158,178,179,158,180, 30,
         31, 32, 33, 30, 34, 28, 35, 36, 37, 38, 30, 39, 40, 41, 42, 43,
         44, 30, 45, 28, 46, 47, 48, 57, 49, 30, 30, 50, 51, 30, 52,  0,
    ],
    iso_8859_4: [
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
          0,159,  0,157, 54,  0,171, 54, 54,172,164,168,  0, 53,180, 54,
         56, 31, 54, 29, 54,  0, 43, 54, 54, 44, 36, 40,  0,  0, 52,  0,
        161,  0,  0,  0,162,156,156,160,165,166,163,  0,167,  0,  0,170,
          0,173,174,169,  0,175,176, 57,156,177,  0,  0,179,  0,178, 30,
         33,  0,  0,  0, 34, 28, 28, 32, 37, 38, 35,  0, 39,  0,  0, 42,
          0, 45, 46, 41,  0, 47, 48, 57, 28, 49,  0,  0, 51,  0, 50,  0,
    ],
    windows_874: [
          3,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,
        255,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,
          0,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
         20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35,
         36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
         52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62,255,255,255,255,  0,
         63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,  0,
          4,  4,  4,  4,  4,  4,  4,  4,  4,  4,  0,  0,255,255,255,255,
    ],
    vietnamese: [
                                                                                                                          0, 58, 58, 22,  0, 33,  7,  0,  0,  0,  0,  1, 57,  3,  0,  0,  0, 39, 14,  8,  3,  0, 11,  0,  5,  0,  0,  0,  0,  0,  0, //  ,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // 0,
                                                                                                                          0,  3, 16,  2,  0,  0,  0,  0,  0,  0,  0,  0,  3,  1,  0,  0, 17,  2,  1,  0,  0,  1,  0,  0,  1,  0,  0,255,255,  0,  0, // a,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // b,
                                                                                                                          0,  0,  0,  0,  0,  0, 16,  0,  0,  0,  0,  0, 51, 12,  0,  0,  0, 48,  0,  0,  0,  0,  2,  0,  0,  0,  0,255,255,  0,  0, // c,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // d,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // e,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // f,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // g,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // h,
                                                                                                                          0, 27, 11,  1,  0, 16,  3,  0,  0,  0,  0,  0, 16,  0,  0,  0, 10, 34,  0,  2,  1,  0,  0,  0,  0,255,  0,255,255,  0,  0, // i,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // j,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // k,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // l,
                                                                                                                          0,  6,  3,  0,  0,  2,  0,  2, 22,  0,  0,  2,  4,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // m,
                                                                                                                          0, 59, 23, 10,  0, 19, 22, 18,  5,  0,  0, 28, 61,  6,  0,  0,  0, 39,  4, 20,  9,  6,  1,  0,  1,  0,  0,255,255,  0,  0, // n,
                                                                                                                          0,  5, 13,  0,  0,  5,  3,  0,  0,  0,  0,  0,  3,  0,  0,  0,  4,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // o,
                                                                                                                          0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0, 13,  0,  0,  0,  0, 10,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // p,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,  0,  0, // q,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // r,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // s,
                                                                                                                          0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0, 25,  0,  0,  0,  0, 40,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // t,
                                                                                                                          0, 12,  3,  1,  0,  1,  0,  4,  0,  0,  0,  6,  4,  0,  0,255,  0,  4,  0,  0,  0,  0,  0,  0,  1,  0,  0,255,255,  0,  0, // u,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // v,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // w,
                                                                                                                          0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,  0,  0, // x,
                                                                                                                          0, 18,  2,  0,  0, 17,  1,  4,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // y,
                                                                                                                          0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, // z,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,255,  0,255,255,255,255,255,255,  0,  0,255,255,  0,255, // œ,
          0,255, 95,255,255,255,  1,255,255,255, 11,255,255,255,255,255,  6,255,255,255,255,255,  8,255,255,255,  4,255,255,255,255,255,255,255,255, 14,  2,255,255, 17,255,255,255,255,255,255,255,  8, 16,255,255,255,  5,255,255,255,255,255,255, // ̀,
          0,255, 39,255,255,255,  0,255,255,255,  5,255,255,255,255,255,  3,255,255,255,255,255, 20,255,255,255,  1,255,255,255,255,255,255,255,255,  3,  0,255,255, 31,255,255,255,255,255,255,255, 10, 10,255,255,255,  5,255,255,255,255,255,255, // ̉,
          0,255, 12,255,255,255,  1,255,255,255,  5,255,255,255,255,255,  0,255,255,255,255,255,  3,255,255,255,  1,255,255,255,255,255,255,255,255,  2,  0,255,255,  3,255,255,255,255,255,255,255,  1,  0,255,255,255,  7,255,255,255,255,255,255, // ̃,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,255,  0,  0,  0,255,255,255,255,  0,255, // ß,
          0,  0,  0,  3,  0,  0,  0,  0, 12, 15,  0,  0,  0, 17,  1,  6, 15,  0,  0,  0,  0,  1,  0, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,255,255,  1,  0,255,255,  0,  0,255,255,  0,255,255,255,255,255,255, // à,
          2,  0,  0,  2, 13,  0,  0,  0,  0, 26,  3,  0,  0,  1,  1,  0,  1,  0,  0,  0,  3,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,255,255,  3,  0,255,  0,  0,255,255,255,  0,255,255,255,255,  0,255, // á,
          3,  0,  0,  1,  7,  5,  0,  0,  1, 23,  0,  0,  0,  3,  3,  0,  0,  0,  0,  3,  1,  7, 12, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255,  8,  0,  0,  0,  0,255,  0,255,  0,255,  0,255,255,255,255, // â,
          0,  0,  0,  3,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1, 23,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,255,  1,  0,255,  0,  0,255,255,255,  0,255,  0,255,255,255,255, // ă,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,255,255,  0,255,255,  0,  0,255,  0,255,255,255,  0,255,255,255,255,255,  0,255, // è,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,  0,255,  0,255,255,255,255,  0,  0,255,255,255,255,255,255,  0,255, // é,
          0,  0,  0,  1,  0,  0,  0,  0,  0, 28, 66,  0,  6,  2,  0,  1,  0,  0,  0,  2,  0,  4,  0,  3,  0,  0, 12,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,255, 13,  0,255,  0,  0,255,255,255,  0,255,  0,255,255,255,255, // ê,
          0,255, 63,255,255,255,  5,255,255,255, 24,255,255,255,255,255, 22,255,255,255,255,255, 18,255,255,255,  3,255,255,255,255,255,255,255,255, 17,  6,255,255, 30,255,255,255,255,255,255,255, 23, 20,255,255,255, 10,255,255,255,255,255,255, // ́,
          0,255,  0,  0,  0,  0,  0,  0,  0, 18,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,  0,255,  0,255,255,255,255,255,255,255,255, // í,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,255,255,255,  0,  0,  0,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,  0,255, // î,
          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,255,  0,255,  0,255,  0,  0,255,255,255,  0,255,255,255,255,255,255,255,255,255,  0,255, // ï,
         97,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,255,  0,255,  0,255,  0,255,255,  0,255, // đ,
          0,255, 42,255,255,255,  0,255,255,255, 15,255,255,255,255,255, 18,255,255,255,255,255,  8,255,255,255,  0,255,255,255,255,255,255,255,255, 34,  4,255,255, 24,255,255,255,255,255,255,255, 41, 17,255,255,255, 17,255,255,255,255,255,255, // ̣,
          0,  0,  0,  2, 10,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255,255,255,  0,  0,  0,255,255,  2,255,  0,  0,255,255,255,255,  0,255,255,255,255,255,255, // ó,
          2,  0,  0,  7,  8,  0,  0,  0,  2, 18,  0,  0,  0,  1, 17,  2,  0,  0,  0,  1,  6,  5, 15,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,255, 22,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,255,255, // ô,
          5,  0,  0,  1,  1,  0,  0,  0,  0,  8,  2,  0,  0,  2,  1,  0,  0,  0,  0,  1,  2,  0,  0,  4,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  2,  0,  0,  0,  0,  0,  0,255, 42,255,  0,255,255,  0,255, // ơ,
          0,255,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,255,255,255,255,255,  0,255,255,255,  0,  0,255,255,255,  0,  0,255,  0,255,  0,  0,255,  0,255,  0,255,255,255,255, // ù,
          0,  0,  0,  0,  1,  0,  0,  0,  0, 14,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,255,255,255,  0,  0,  0,255,255,  0,255,  0,255,255,  0,  0,255,  0,255,255,255,255,255,255, // ú,
          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255,255,255,255,255,255,255,  0,  0,255,  0,255,255,255,  0,255,255,255,  0,255,255,255,255,255,255,255, // û,
          1,  0,  0,  1,  4,  4,  0,  0,  7, 28,  1,  0,  0,  4,  0,  3,  0,  0,  0,  4,  7,  9,  0,  2,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0,  0,255,255, 15,  0,255,  0,  0,255,  0,255,  0,255,  0,255,255,255,255, // ư,
          0,255,  0,255,255,  0,255,  0,255,255,255,255,255,  0,255,255,  0,255,255,  0,255,255,  0,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,  0,255,255,255,255,255, // ÿ,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,255,255,  0,  0,  0,255,255,  0,  0,255,  0,255,255,255,255,  0,255,  0,  0,  0,  0,  0, // ·,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,255,255,255,255,255,255,255,255,  0,255,255,255,255,  0,255,255,255,255,255,255,255,255,  0,255,  0,  0,  0, // ¨,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,  0,255,255,255,  0,  0,  0,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,  0,255,  0,  0, // ®,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,255,255,255,255,255,255,255,255,  0,255,255,255,255,  0,255,255,255,255,255,255,255,255,  0,  0,  0,255,  0, // ©,
          0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,255, // ¬,
      //   ,  0,  a,  b,  c,  d,  e,  f,  g,  h,  i,  j,  k,  l,  m,  n,  o,  p,  q,  r,  s,  t,  u,  v,  w,  x,  y,  z,  œ,  ̀,  ̉,  ̃,  ß,  à,  á,  â,  ă,  è,  é,  ê,  ́,  í,  î,  ï,  đ,  ̣,  ó,  ô,  ơ,  ù,  ú,  û,  ư,  ÿ,  ·,  ¨,  ®,  ©,  ¬,
    ],
    central: [
                                                                                                                         28,  7, 25, 20,  2,102, 51, 12, 10, 15,  0,  0, 69,  0,254,  0,  0, 52,  0, 45, 73, 79,  0, 35,159,  9,  2,  3, 11,  4, 43,  0, 33,  0,  5, 22, 17,  5,  1, 52,  5,  4,  0,  0,  0, //  ,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                                                                                                                         34,  0,  5, 97,  0, 76,  0,  7, 11,  8,  0,  0,  0,  0,  1,  0,  0, 39,  0, 69,  0,  0,  0,  0,  7,  0,  5, 17,  0,  4,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0, 32,  0,255,255,  0, // a,
                                                                                                                          0,  0,  0,  9,  0,  0,  1,  0,  2,  1,  0,  0, 19,  0,  0,  0,  0,  0,  0,  1,  6,  3,  0,  0,  1,  0,  0,  0,  0,  0,  4,  0,  2,  3,  0,  0,  5,  0,  0,  0,  0,  0,255,255,  0, // b,
                                                                                                                          1, 41,  0,  0,  0,  6, 30,  6,  0,  3,  0,  0, 23,  0,  5,  0,  0,  0,  0,  0,  2, 20,  0,  2, 21,  2,  0,  0,  9,  0,  4,  0,  0,  0,  0,  0,  6,  0,  0, 24,  0,  0,255,255,  0, // c,
                                                                                                                          0,  0,  0,  5,  4,  0, 12,  0,  0,  1,  0,  0, 11,  0,  4,  0,  0,  0,  0,  0,  6, 18,  0,  3,  5,  0,  0,  0,  0,  0, 22,  0,  2,  1,  0,  0,  5,  0,  0,  0,  0,  0,255,255,  0, // d,
                                                                                                                         45,  0,  0, 71,  0, 10,  0, 18,  0, 24,  0,  0,  0,  0,  0,  0,  0, 40,  0, 93,  3,  0,  0,  0,  0,  0,  0, 66,  0,  0,  0,  0,  1,  0, 42,  0,  0,  0,  0,  0, 22,  0,255,255,  0, // e,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // f,
                                                                                                                          0,  0,  0,  1,  0,  1,  3,  0,  0,  0,  0,  0, 38,  0,  1,  0,  0,  0,  0,  0, 35,  6,  0,  0,  2,  0,  0,  0,  0,  0,  4,  0,  0,  1,  0,  0,  8,  0,  2,  0,  0,  0,255,255,  0, // g,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0, 19,  0,  0,  2,  7,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2,  0,  0,255,255,  0, // h,
                                                                                                                         72,  0,  0, 51,  0,  0,  0,104,  0,  0,  0,  0,  1,  0,  4,  0,  0,111,  0,114,  0,  0,  0,  0,  0,  4,  0,  4,  0,  0,  3,  0,  3,  0, 16,  0,  0,  0,  0,  0,160,  0,255,255,  0, // i,
                                                                                                                          4,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,  0, 17,  0,  0,  0,  6,  6,  0,  0,  0,  0,  0,  4,  0,  1,  0,  0,  0,  5,  0,  0,  0,  0,  0,255,255,  0, // j,
                                                                                                                        124,  0,  0,104,  0, 13,  0,  0,  6,  5,  0,  0, 12,  0,  0,  0,  0,  0,  0, 92, 24,  5,  0,  5, 13,  3,  0,  0,  0,  0,  7,  0,  5,  5,  1,  0,  0,  1,  3,  1,  1,  0,255,255,  0, // k,
                                                                                                                          6, 10,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0, 54,  2,  8,  0,  0,  0,  0, 20, 28,  0,  0, 10,  6,  3,  0,  0,  0,  0, 14,  0,  8,  9,  0,  1,  4,  0, 33,  1,  0,  0,255,255,  0, // l,
                                                                                                                          1,  6,  0,  1,  0,  1,  0,  0,  1,  0,  0,  0, 13,  3, 10,  0,  0,  0,  0,  0, 10,  0,  0,  5, 18,  7,  0,  0,  0,  0,  2,  0,  0,  1,  0,  2,  2,  0,  0, 11,  0,  0,255,255,  0, // m,
                                                                                                                         24,  9,  0, 25,  4,  9,  0,  1,  2, 13,  0,  0, 44, 60,  6,  0,  0,  4,  0, 78, 32,  0,  0, 11, 10,158,  0,  0,  0,  0, 18,  1,  1, 10,  0,  0,  2,  1,  2,  0,  0,  0,255,255,  0, // n,
                                                                                                                         10,  0,  1,  3,  0, 61,  0,  5, 10,  8,  0,  0,  0,  0,  0,  0,  0,  1,  0,  5,  2,  0,  0,  0,  3,  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // o,
                                                                                                                         14,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  4,  0,  2,  0,  0,  0,  0,  0, 26, 25,  0,  2,  1,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  3,  0,  0,  0,  0,  0,255,255,  0, // p,
                                                                                                                          0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // q,
                                                                                                                          0,  7,  0,  1,  3,  2,  0,  0,  0,  0,  0,  0, 56,  9, 46,  0,  0,  0,  0,  9, 40,  0,  0,  5, 11,  3,  0,  0,  0,  0, 29,  1,  3, 16,  0,  0,  6,  2,  2,  1,  0,  0,255,255,  0, // r,
                                                                                                                          0,  0,  0,  5,  0,  0,  4,  0,  4,  2,  0,  0, 78,  0, 10,  0,  0,  0,  0,  1, 86,  6,  0,  7, 18,  1,  0,  0, 44,  0,  6,  2,  8,  7,  2,  3,  4,  3,  1,  2,  0,  0,255,255,  0, // s,
                                                                                                                        182,  0,  0,  0,  0,  2,  6, 49,  0,  0,  0,  0, 37,  5, 30,  2,  0,  0,  0,  4, 17, 15,  0, 14, 26,  1,  0,  0,  0,  0,  3,  1,  3,  8,  0,  0,  4,  0,  2,  1,  0,  0,255,255,  0, // t,
                                                                                                                          5,  0,  0, 23,  0, 23,  0,  6,  3,  3,  0,  0,  0,  6, 11,  0,  0, 10,  0, 12,  0,  0,  0,  0,  0,  0,  0, 22,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,255,255,  0, // u,
                                                                                                                          9,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 11,  0,  1,  0,  0,  0,  0,  4, 22,  0,  0,  1,  6,  0,  0,  0,  0,  0,  1,  4,  3,  8,  0,  4,  1,  4,  0,  4,  0,  0,255,255,  0, // v,
                                                                                                                          0, 22,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0, 75,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // w,
                                                                                                                          0,  0,  0,  0,255,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,  0,  0,  2,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // x,
                                                                                                                          0,  0,  0,  0,  0, 24,  0,  0,  0, 24,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // y,
                                                                                                                          0,  0,  0,  0,  0,  0,  8,  0,  0,  0,  0,  0, 23,  0,  6,  1,  0,  0,  0,  0,  4, 19,  0,  3,  3,  0,  0,  0,  0,  0,  2,  0,  5, 18,  0,  1, 21,  0,  1,  2,  0,  0,255,255,  0, // z,
        116,  0, 57,  1,  0,  3, 45,  0, 15,  0, 82, 26,  2,  3,  9,  7, 66, 83,  0, 40,  0,  2, 24,  5,  0,  0,  6,  3,  4,  0,  0,  4,255,  0,  0,  0,  0,255,255,  0,  3,  0,  0,  0,  0,  0,255,  1,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,255,  0,255,255,  0, // š,
         35,  0,  4,  0,  0,  0, 16,  0,  0,  0,  4,  2,  0,  0,  0,  0, 58,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,  4,255,  0,  0,255,255,255,  0,255,  0,  0,255,255,  0,255,  0,255,  0,  0,255,  0,255,255,255, // ś,
          0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,255,  0, 14,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,255,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,  1,255,  0,255,255,  0,  0,  0,255,255,  0,  0,  1,255,  0,  0,  0,  0,255,255,255, // ť,
         61,  0, 22,  0,  0, 12, 44,  0,  0,  0, 19,  0,  0,  3,  0,  2,110,  0,  0, 84,  0,  0, 44,  0,  0,  0,  1,  0,  0,255,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  2,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  3,  3,  0,  0,  0,  0,  0,  0,  2,255,  0,  0,  1,  0,  0,  0,  0,255,  0,255,255,  0, // ž,
          3,  0,  5,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,255,255,  0,  0,  0,255,  0,  0,255,255,  0,  0,255,  0,  0,255,  0,255,  0,255,  2,  0,255,255,255,  0,255,255,255,  0,255,  0,255,255,255, // ź,
         12,  0, 85,  2,  2, 14, 14,  0, 15,  1, 28,  0, 10,  0,  2,  0, 26,  7,  0,  1, 27,  1, 37,  0, 12,  0, 21,  9,  0,  0,255,  0,  0,  0,  3,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  0,  0,  0,255,255,  0,  0,255, 20,  0,255,  0,255,  0,  0,255,255,255,255,  0,255,255,255, // ł,
          0,  0,  0,  1,  3,  2,  0,  0,  1,  0, 25, 29,  6,  5,  0, 10,  0,  0,  0,  4,  6,  3,  0,  0,  5,  0,  0, 18,  0,  0,255,  0,  0,  5,  0,  0,  0,  1,255,  0,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,255,  0,  0,  0,  0,255,255,255, // ą,
         92,  0, 31,  0,  0,  0, 33,  0,  0,  0, 22,  0,  0,  0,  0,  0, 12,  0,  0,  3,  0,  0,  7,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,  0,255,255,255,255,  0,  1,  2,  0,255,255,  0,255,  0,  0,255,255,  0,  3,255,255,255,  0,  0,255,  0,  0,255,  0,  0,255,  0,255,  0,  0,255,255,  0, // ş,
          4,  0,  1,  0,  0,  3, 28,  0,  0,  2,  0,  0,  0,  0,  0,  0,  6,  0,255,  0,  0,  0,  1,  0,255,255,  0,  0,  0,255,  0,  0,255,255,255,255,  0,255,255,  0,  2,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,255,  0,255,255,255,  0,  0,255,  0,255,  0,  0,255,  0,  0,255,  0,255,255,  0, // ľ,
         19,  0, 10,  0,  0,  2, 17,255,  0,  0,  3,  0,  3,  0,  0,  0, 18,  0,  0,  0,  0,  0, 16,  0,  0,  0,  6,  0,255,  0,255,  0,  0,  0,  7,255,255,  0,255,255,  0,  0,255,  0,255,  0,255,255,  0,  6,  0,  0,255,255,  0,255,  0,255,  3,  0,  0,255,255,  0,255,255,255,255,255,  0,255,255,255, // ż,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255,255,255,255,255,255,  0,255,  0,255,255,  0,255,255,255,255,  0,255,  0,255,  0,255,255,255,255,255,255,  0,  0,  0,  0,255,255,255,  0,255,255,255,255,255,255, // ß,
          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,255,255,255,255,255,255,255,  0,  0,255,255,  0,  0,255,255,  0,  0,255,255,  0,  0,  0,255,255,255,255,  0,255,255,  0,255,255,  0,255,255,255,255,255,255,255,255, // ŕ,
         22,  0,  0,  6,  5, 14,  1,  4, 12, 12,  6, 20, 24, 36, 30, 57,  0,  9,  0, 48, 29, 51,  4, 37,  0,  0,  8, 48,  0,  0,  0,  2,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  4,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // á,
          0,  0,  0,  1, 12,  1,  0,  3,  1,  3,  0,  0,  0,  1, 32,  0,  0,  5,  0, 11,  1,  8,  0,  6,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,255,  0,255,  0,255,  0,255,255,255,255,  0,255,255,255,  0,255,  0,  0,255,255,255, // â,
          0,  0,  0,  9, 70,  9,  0,  6, 13,  1,  0,  0,  0, 34, 18, 41,  0, 23,  0, 48, 43, 88,  5,  8,  0,  1,  0, 19,  0,  0,255,  0,255,  0,255,  0,255,  0,  0,255,  0,  0,  0,255,255,255,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255, 23,  0,255,255,255, // ă,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,255,  0,  0,255,  0,255,255,255, // ä,
          0,255,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,255,  0,255,255,255,255,255,255,  0,255,255,  0,  0,255,255,  0,  0,  0,255,  0,  0,255,  0,255,255,255,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255, // ĺ,
          8,  0, 12,  0,  0,  0, 48,  0,  0,  0, 63,  0,  1,  0,  0,  0, 14, 77,  0,  0,  0,  0, 24,  0,  0,  0,  3,  0,  9, 27,255,  0,  0,  0,  0,255,255,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  1,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255,  0,  0,  0,255,  0,  0,255,  0,255,255,255, // ć,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,255,255,255,255,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,255,  0,  0,255,255,255,255,255,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,255, // ç,
         76,  0,108, 20,  0,  0, 56,  0,  0,  1, 93,  2,  0,  2,  7, 17, 68,  1,  0,  7,  2,  0, 36,  3,  0,  0,  1,  0, 49,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  3,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  2,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  4,255,  0,  0,  0,  0,255,255,  0, // č,
         62,  0,  0,  3,  1,  9,  0,  6,  8,  2,  1,  3, 40, 29, 24, 43,  0,  6,  0, 14, 29, 43,  0, 19,  0,  0,  5, 19,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // é,
          0,  0,  0,  2,  1,  1,  0,  0,  3,  0, 67, 35,  6,  3,  5, 21,  0,  1,  0,  9,  1, 27,  0,  0,  5,  0,  0, 17,  0,  0,255,  0,  0,  3,  0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,255,255,255,  0,255,  0,255,255,  0,  0,  0,255,255,255, // ę,
          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,255,255,  0,255,255,  0,255,  0,255,255,  0,255,255,  0,255,255,255,255,  0,255,255,  0, // ë,
          0,  0,  0,  7,  0, 13,  0,  0,  0,  0,  0,  0,  0,  0, 24, 35,  0,  4,  0,  0,  0, 15,  0, 24,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,255,255,255,  0,255,255,255,255,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,255,255,255,  0,  0,255,255,255, // ě,
          6,  0,  1,  4, 16,  9,  1,  0,  1,  3,  1, 14,  3,  8, 14,117,  0,  4,  0, 19,  6, 18,  1, 17,  0,  0,  4,  8, 15,  0,  0,  5,  0,  0,  0,  0,  0,  0,255,  0,  2,255,  0,  0,  0,  0,  0,  7,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 34,  0,  0,  0,  0,  0,  0,  0,255,255,255, // í,
        158,  0,  0,  1,  0,  1,  0,  0,  0,  0,  2,  1,  6,  5,  5,  5,  0,  0,  0,  4,  2,  3,  0,  3,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  1,255,255,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255,  0,255,255,255,255,255,255,255,255,  0,  0,255,255,255, // î,
          2,255,  2,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255,255,255,255,255,  0,255,255,  0,255,  0,255,  0,  0,255,255,  0,  0,255,  0,255,255,  0,  0,255,255,255,  0,  0,  0,255,255,  0,255,  0,255,255,255, // ď,
          3,  0, 20,  0,  0,  0, 27,  0,  0,  0,  1,  0,  0,  0,  0,  1, 34,  0,  0,  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,255,255, 31,255,255,255,255,255,255,255,255,  0,  0,  0,  0,255,  0,255,  0,  0,255,255,  0,  0,255,  0,  0,255,255,  0,255,255,255,255,255,  0,255,255,255,255,  0,255,255,  0, // đ,
          0,  0, 31,  0,  0,  0, 10,  0,  0,  0,  8,  0,  0,  0,  0,  0,  8,  0,  0,  0,  0,  0,  1,  0,  0,255,  4,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,255,  0,255,  0,  0,255,  0,255,255,  0,255,255,255, // ń,
          0,  0,  0,  0,  0,  1,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,255,255,255,255,  0,255,255,  0,  0,255,  0,  0,  0,255,255,  0,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255, // ň,
          2,  0,  0,  4,  6,  5,  1,  1, 26,  2, 10,  3, 15, 25,  3,  5,  0, 17,  0, 34,  4, 21,  0,  1, 20,  0,  1, 12,  0,  0,  0,  0,  0, 11,  0,  0,  0,  1,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // ó,
          0,  0,  0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  2,  0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,  0,  0,  0,255,  0,255,255,  0,255,255,  0,  0,  0,255,255,255,  0,  0,255,  0,255,  0,  0,  0,255,  0,255,  0,  0,255,255,255,  0,  0,255,255,255, // ô,
          2,  0,  0,  3,  0,  7,  0,  4,  0,  0,  0,  0,  3, 13,  0,  5,  0,  0,255,  5,  8, 19,  0,  1,  0,  0,  2, 13,255,255,255,  0,255,  0,255,  0,255,255,  0,255,  0,255,255,255,255,255,  0,255,  0,255,255,255,  0,255,255,255,255,255,  0,255,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255,255, // ő,
          7,  0,  0,  1,  0,  3,  0,  3,  1,  1,  0,  1, 33,  4,  0,  3,  0,  0,  0,  5,  1, 13,  0,  1,  0,  0,  1,  9,  0,  0,  0,  0,255,  0,255,  0,255,  0,  0,255,  0,255,255,  0,255,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,255,  0, // ö,
         11,  0,  4,  3,  0,  2,  2,  0,  0,  1,  1,  0,  2,  0,  0,  0,  8, 44,  0,  0,  0, 12,  1,  1,  0,  0,  1,  0,  0,255,255,  0,255,255,255,255,255,255,255,  0,  6,  0,255,  0,255,255,  0,  0,  0,255,  0,  2,  3,255,255,255,255,  0,  0,255,255,  0,  0,  0,  1,255,255,  0,255,  0,255,255,255, // ř,
          0,  0,  0,  0,  1,  3,  0,  0,  0,  1,  0,  1,  3,  2,  2,  2,  0,  4,  0,  6,  0,  2,  0,  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,255,255,255,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255, // ů,
         15,  0,  0,  0, 18,  1,  0,  0,  0,  0,  2, 13,  4,  2,  1,  4,  0,  0,  0,  8, 19,  7,  0,  1,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255, // ú,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 10,  0,  0,  0,  0,  2,  1,  1,  0,  0,  0,  0,  1,  0,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,  0,255,255,255,  0,255,  0,255,255,255,  0,255,  0,255,255,  0,  0,  0,  0,255,255,255,255,255,255, // ű,
          3,  0,  0,  1,  0,  0,  0,  2,  1,  1,  0,  0,  5,  1,  1,  0,  0,  8,  0,  8,  2,  2,  0,  0,  0,  0,  1,  8,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,255,  0,255,255,  0,255,  0,  0,  0,  0,255,  0,255,  0,255,  0,255,  0,  0,  0,255,  0,  0,255,255,  0,  0,  0,  0,255,  0,255,255,255, // ü,
          0,  0,  0,  2,  0,  1,  0,  0,  0,  2,  0,  0, 26,  2,  0, 34,  0,  0,  0,  6,  0,  5,  0, 28,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,255,255,  0,  0,255,255,  0,255,  0,255,  0,  0,255,  0,  0,  0,255,255,255,255,  0,255,  0,255,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255, // ý,
         13,  0, 73,  1, 22,  0, 20,  0,  0,  0, 20,  0,  0,  3,  1, 51,  3,  1,  0,  6,  0,  0, 16,  1,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,  0,255,255,255,  0,  0, 16,  0,255,255,255,255,  0,  0,255,  0,  0,  2,255,255,  0,255,  0,  0,255,255,255,  0,  0,255,255,255,  0,  0,255,255,255, // ţ,
          4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ·,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,  0, // ¨,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,  0,255,255,255,255,255,  0,  0,  0,  0,255,255,255,255,  0,255,255,255,  0,255,255,255,255,255,  0,255,255,255,255,255,255,255,  0,255,255,255,255,255,  0, // ®,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,255, // ©,
      //   ,  0,  a,  b,  c,  d,  e,  f,  g,  h,  i,  j,  k,  l,  m,  n,  o,  p,  q,  r,  s,  t,  u,  v,  w,  x,  y,  z,  š,  ś,  ť,  ž,  ź,  ł,  ą,  ş,  ľ,  ż,  ß,  ŕ,  á,  â,  ă,  ä,  ĺ,  ć,  ç,  č,  é,  ę,  ë,  ě,  í,  î,  ď,  đ,  ń,  ň,  ó,  ô,  ő,  ö,  ř,  ů,  ú,  ű,  ü,  ý,  ţ,  ·,  ¨,  ®,  ©,
    ],
    cyrillic: [
                      0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0, 16, 38,  0,  0,  2,  5,  0, 10,121,  4, 20, 25, 26, 53,  9,  5, 61, 23, 20, 26, 15, 95, 60,  2, 25, 15, 25, 29,  0, 14,  6,  6, 25,  1,  0, 27, 25,  8,  5, 39, //  ,
                      0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                      0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // a,
          0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ·,
          0,  0,255,  0,255,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¨,
          0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ®,
          1,  0,255,  0,  0,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ©,
          0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¬,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,255,255,  0, // ѓ,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,255,  0,255,255,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255, // ђ,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,  0, // љ,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,255,  5,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,255,255, // њ,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0, // ќ,
          0,  0,  0,  0,255,255,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,255,  0,  5,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,255,  0, // ћ,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,255,255, // џ,
          7,  0,  0,  0,255,255,  0,255,255,255,255,  0,  0,  0,255,  0,  1,  0,  0,  0,  0,255,  0, 15,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1, // ў,
         12,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2,  3, 14,  5,  5,  0,  0,  4,  0,  0, 21, 15, 10, 17,  0,  6, 14,  4,  6,  0,  3,  1,  8,  1,  0,  0,  0,  2,  0,  0,  0,  0, // і,
          0,  0,  0,  0,255,255,  0,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0, // ґ,
          0,  0,  0,  0,255,255,  0,255,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,255,255,  0,  4,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ё,
          6,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  5,  0,  0,  0,  1,  7,  0,  3,  2,  0,  0,  0,  0,  0,  0,  2,  2,  2,  0,  0,  2,  2,  5,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // є,
         12,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  5,  1,  0,  0,  0,  2,  0,  0, 20,  0,  0,  0,  0,  0, 15,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0, // ј,
          0,  0,  0,  0,255,255,  0,255,  0,255,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0, // ѕ,
          9,  0,  0,  0,255,255,  0,255,255,  0,255,255,255,255,255,  0,  5,  0,  0,  0,  0,  0, 13,  3,  3,  0,  4,  1,  0,  1,  2,  0,  0,  0,  1,  0,  0,  4,  0,  0,  1,  3,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ї,
         31,  0,  0,  0,255,255,  0,255,  0,  2,  2,  2,  0,  0,  0,  0,  1,  0,  0,  0, 27,  0,  0, 23, 22, 26, 22, 19,  0,  3, 11,  5,  0, 43, 38, 18, 58,  1, 21, 44, 17, 54,  1,  2, 28,  5,  8,  3,  1,  9,  0, 12,  0,  0,  0, // а,
         39,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  1,  7,  0,  1,  1,  0,  0,  7,  4,  1,  9,  0,  1,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, // б,
         31,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  3,  0,  1,  0, 18,  0,  0,  1,  1,  6,  0,  2,  6,  0,  1,  0,  1,  0, 32,  0,  2,  2, 23,  9,  0,  0,  0,  1,  0,  0,  1,  1,  0,  3,  0,  2, // в,
         23,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  7,  0,  0,  1, 20,  0,  0,  1,  0,  9,  0,  0,  9,  7,  0,  5,  2, 18, 11,  0,  8,  3,  2,  3,  0,  0,  0,  0,  0,  0,  0,  3,  0, 13,  0,  3, // г,
         26,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  9,  0,  0,  2,  0,  0,  2, 19,  0,  1,  5,  0, 13,  2,  2,  3,  2,  0,  6,  1, 12, 30,  0,  4,  0,  0,  7,  0,  0,  0,  0,  0,  0,  1,  0,  0,  5,  0,  1, // д,
         12,  0,  0,  0,255,255,  0,255,  0,  1,  4,  5,  1,  0,  0,  0,  0,  0,  0,  0, 24,  0,  1,  5,  7, 11,  3, 12,  1,  6,  6, 10,  0,  3, 15, 14, 14,  4,  8, 25, 14, 29,  0,  1,  1,  4,  8,  8,  2,  0,  3,  1,  0,  0,  0, // е,
          6,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1,  2,  2,  0,  0,  0,  0,  0,  3,  2,  1,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, // ж,
         19,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  1,  6,  0,  0,  0, 11,  8,  0,  0,  8,  0,  0,  0,  0,  0,  4,  0,  1,  0,  0,  3,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1, // з,
         24,  0,  0,  0,255,255,  0,255,  0,  0,  0,  1,  0,  5,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1, 10, 16, 21, 22,  0,  6,  5,  6,  1, 15, 15,  8, 38,  2,  4, 27,  9, 15,  0,  3,  8, 12,  7,  6,  1,  0,  0,  0,  0,  0,  0, // и,
          6,  0,  0,  0,255,255,  0,255,255,  0,255,255,255,  0,255,  0,  7,  0,  0,  0,  0,  0,  4, 21,  0,  0,  0,  0,  5,  0,  0, 39,  0,  0,  0,  0,  0,  8,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  5,  0,  3,  0,  0, // й,
         54,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  1,  8,  0,  0,  0,  0,  0,  0, 10,  0,  1,  0,  1, 11,  0,  0, 12,  0,  1,  2,  0,  3,  8,  0,  2, 23,  2,  4,  0,  2,  3,  3,  8,  0,  0,  3, 16,  1,  4,  3, // к,
         11,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  2,  6,  0,  0,  6,  0,  0,  4, 29, 12,  4,  5,  2, 18,  0,  0, 17,  4,  5, 11,  0,  0, 21,  2,  3,  4,  1, 15,  1,  0,  0,  0,  0,  0,  4,  3,  2, 12,  0,  2, // л,
         23,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  4,  0,  0,  0, 17,  1,  0,  0,  0,  7,  0,  1, 13,  2,  0,  0,  0,  0, 13,  0,  2,  4,  0,  2,  0,  0,  0,  0,  0,  0,  1,  4,  2,  4,  1,  1, // м,
         42,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  4, 12,  0,  6,  7,  1,  0,  7, 76,  0, 22,  1,  4, 27,  1,  3, 34, 30,  0,  7,  1, 13, 24,  1,  3,  5,  3,  4,  0,  1,  0,  4,  1,  0,  2, 18,  7, 16,  0,  4, // н,
         37,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1,  0,  0,  1, 10, 27, 22, 15,  1,  2,  3,  7,  5, 32, 11,  7, 38,  8, 21, 23, 11, 23,  0,  2, 10,  2,  2,  3,  2,  0,  0,  1,  0,  0,  0, // о,
         47,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  1,  0,  0,  2,  0,  1,  2,  4,  0,  0,  2,  0,  6,  0,  0,  5,  0,  2,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0, // п,
         19,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  8,  0,  0,  5, 47,  4,  6,  6,  5, 22,  0,  0,  5,  2,  6,  0,  0,  0, 22, 22,  0,  1, 14,  9,  1,  0,  1,  0,  0,  0,  7,  2,  8, 16,  0,  3, // р,
         53,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  4,  9,  0,  2,  0,  1,  0,  2, 21,  1,  4,  1,  2, 11,  0,  0, 12,  2,  4,  7,  1, 13, 15,  1,  4,  6,  3,  6,  0,  0,  0,  0,  0,  0,  1,  2,  3,  5,  0,  1, // с,
         28,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  1,  6,  0,  0,  1,  0,  0,  1, 32,  0,  1,  3,  0, 12,  0,  1, 22,  1,  3,  7,  1,  6, 23,  0, 14, 41, 14,  3,  0,  1,  1,  1, 21,  0,  2,  2,  6,  2,  1,  4, // т,
         15,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2,  4,  2,  4,  6,  3,  0,  2,  0,  0,  6,  5,  6,  3,  0,  3,  7,  4,  7, 18,  1,  6,  0,  2,  0,  0,  0,  0,  0,  0,  1,  0, // у,
          8,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ф,
         41,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2, 30,  0,  2,  0,  0, 11,  0,  0,  5,  1, 14,  3,  0,  3,  6,  0,  7,  0,  0,  1,  0,  1,  0,  2,  0,  0,  0,  4,  3,  5,  0,  0, // х,
          8,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  4,  0,  0,  7,  1,  0,  1,  0,  2,  1,  0,  0,  9,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  1,  1, // ц,
          6,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  5,  0,  1,  5,  0,  2,  0,  0,  6,  0,  0,  1,  0,  0,  3,  0,  2,  0,  0,  2,  0,  1,  0,  0,  3,  0,  0,  2,  0,  0,  0,  0, // ч,
         12,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, 17,  0,  0,  1,  0,  2,  0,  0, 26,  0,  0,  0,  0,  0, 22,  2,  6,  0,  0,  5,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  0,  0, // ш,
          2,  0,  0,  0,255,255,  0,255,255,  0,255,255,255,255,255,  0,  0,  0,  0,  0,255,255,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, // щ,
          0,  0,  0,  0,255,255,  0,255,255,255,255,255,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0,  0,  3,  4,  0,  2,  0,  0,  0,  0,  0, 11,  0,  1,  0,  0,  2,  2,  5,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ъ,
          1,  0,  0,  0,255,255,  0,255,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, 11,  0,  4,  0,  2,  1,  0,  0,  0,  3,  1, 16,  0,  0, 22,  2, 10,  0,  0,  0,  8,  6,  3,  0,  0,  0,  0,  0,  0,  0, // ы,
          0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, 15,  0,  7,  3,  0,  1, 13,  7,  7,  0, 35,  6,  0,  0,  0,  0,  0,  0,  0,  6,  0, // ь,
         10,  0,  0,  0,255,255,  0,255,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  1, 10, 11,  0,  2,  2,  0,  0,  0,  9,  3,  9,  0,  0,  7,  6,  9,  0,  0,  8,  3,  2,  1,  0,  0,  0,  0, 17,  0,  0, // э,
         14,  0,  0,  0,255,255,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ю,
          5,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,255,  0,255,  0,  9,  0,  0,  0,  0,  0,  0, 11,  0,  3,  0,  0,  0,  0,  2, 24,  0,  0,  5,  1, 13,  1,  0,  2,  3,  1,  0,  0,  1,  3,  0,  0,  0,  0, 16,  1,  0,  0,  0, // я,
      //   ,  0,  a,  ·,  ¨,  ®,  ©,  ¬,  ѓ,  ђ,  љ,  њ,  ќ,  ћ,  џ,  ў,  і,  ґ,  ё,  є,  ј,  ѕ,  ї,  а,  б,  в,  г,  д,  е,  ж,  з,  и,  й,  к,  л,  м,  н,  о,  п,  р,  с,  т,  у,  ф,  х,  ц,  ч,  ш,  щ,  ъ,  ы,  ь,  э,  ю,  я,
    ],
    western: [
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, //  ,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                                                                                                                          2,  0,  0,  0,  0,  0,  0,  5,  2,  0, 82,  9, 76,  0,  0,  0,231,  0,  0, 48,  0,  0,  0,  0, 11,  2,  0, 33,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // a,
                                                                                                                          0,  0, 19,  5,  5,  0,  0,  2, 11,  4,  0, 34, 22,  0,  0,  0,  4,  0,  0,  0,  6,  1,  3,  0, 41, 31,  8,  7,  0, 66,255,  0,  0,  2,255,255,  0,  0,255, // b,
                                                                                                                          0,  0,  0,  9,  6,  1,  0, 21, 10,  0,  0, 18, 52,  1,  0,  1, 17,  3,  1,  2, 40,  7,  0,  0,  6,  0,  3,  5,  1, 34,255,  0,  0,  0,255,255,  3,  0,255, // c,
                                                                                                                          0,  0,  0,  5,  5,  0,  0, 12, 44, 13,  1,  6, 41,  0, 13,  3, 10,  0,  2,  0, 66, 10,  5,  0, 32, 80,  3,  4,  0, 19,255,  0,  0,  0,255,255,  0,  0,255, // d,
                                                                                                                          2,  0, 61,  0,  0,  0,  2, 30, 12,  1,  3,  0, 85,  0,  0,  0,  0,  0,  1, 18,  0,  0,  0, 48,  0,  9,  0,  3,  0,  9,255,  1,  0,  0,255,255,  0,  0,255, // e,
                                                                                                                          0,  0,  0,  8,  8,  0,  0, 10,  1,  6,  0,161, 22,  0, 13,  0,  4,  0,  0,  0,  1,  3,  0,  0, 15,  3,  0,  0,  0,  4,255,  0,  0,  0,255,255,  0,  0,255, // f,
                                                                                                                          0,  0,  1, 13, 15, 20,  0, 28,  8, 34,  0, 13, 84,  0, 14,  8,  3,  0,  0,  0, 15,  8,  7,  0, 12, 27, 13,  0,  0, 12,255,  0,  0,  0,255,255,  0,  0,255, // g,
                                                                                                                          0,  0,  0,  0,  0,  0,  0, 47,  2,  0,  0,  0,  1,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0, 19,  0,  0,  0,  0, 39,255,  0,  0,  0,255,255,  0,  0,255, // h,
                                                                                                                          1,  0,  5,163,119,  0,  0,141,  0,  2,  3, 87,254,  0,  0,  0,  0,  0,  0,  3, 28,107,  0,  0,  8,  1, 43, 32,  0,  3,255,  2,  0,  0,255,255,  0,  0,255, // i,
                                                                                                                          0,  0,  0,  0,  0,  0,  0, 39,  9,  0,  0,  2,  1,  0,  2,  0,  0,  0,  0,  4,  0,  0,  0,  0, 18, 36,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // j,
                                                                                                                          1,  0,  0,  0,  1,  0,  0, 47, 22, 26,  1, 59,  0,  0,  4,  0,  1,  0,  0,  0,  1,  3,  0,  0, 20, 36,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // k,
                                                                                                                          0,  0,  4, 12, 16,  0,  0, 53, 29, 40,  0, 63, 35,  0, 39,  6, 12,  3,  0,  0, 27,  9,  3,  0, 41, 24,  2,  9,  7, 11,255,  0,  0,  4,255,255,  0,  0,255, // l,
                                                                                                                          0,  0,  0, 14,  5,  3,  0, 58, 11,  4,  3, 48, 29,  6, 28,  1, 10,  1,  0,  0, 24, 40,  2,  0, 18, 18,  1,  7,  2,  3,255,  0,  0,  0,255,255,  0,  0,255, // m,
                                                                                                                          0,  0,  1, 24, 87, 16,  0,131, 61, 60,  1, 54, 69, 32, 77,  7, 34,  2,  3,  0, 93,254, 10,  0, 32, 31, 24, 20,  1, 60,255,  0,  0,  0,255,255,  0,  0,255, // n,
                                                                                                                          1,  0,  0,  0,  0,  0,254,  6,  0,  0, 27,  0, 13,  0,  0, 83,126,  0,  0, 59,  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // o,
                                                                                                                          0,  0,  0,  5,  2,  0,  0,  9, 11,  0,  0,  3, 33,  0,  6,  0,  6,  0,  0,  0, 20, 12,  9,  0, 10, 16,  0,  3,  0,  1,255,  0,  0,  0,255,255,  0,  0,255, // p,
                                                                                                                          0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1, 28,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // q,
                                                                                                                          0,  0,  0, 81, 60,  1,  0,193,105,102,  0,219, 92, 52,189, 37, 18,  1,  0,  0,173, 23, 16,  0,188,137, 13, 12,  2,129,255,  0,  0,  0,255,255,  0,  0,255, // r,
                                                                                                                          0,  0,  1, 40, 33,  0,  0, 41, 24, 34,  0, 67,108, 15,158,  5, 41, 19,  4, 56, 13, 17,  1,  0, 47, 31,  4, 11,  9, 15,255,  0,  0,  0,255,255,  0,  0,255, // s,
                                                                                                                          1,  0,  7, 13, 20,  7,  0, 54, 27, 25,  0,102, 65, 14,111,  3, 48,  9,  4,  0, 89,  7, 19,  0, 13, 11, 39, 12, 15, 18,255,  0,  0,  0,255,255,  0,  0,255, // t,
                                                                                                                          0,  5,  0,  1,  2,  0,  0, 30,  0,  0,  1, 14,  2,  0,  1,  0,  1,  0,  0,  1,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // u,
                                                                                                                          0,  0,  0,  1,  6,  0,  0, 29, 24, 10,  0, 19, 45,  0, 15,  0,  7,  0,  1, 30,  2,  1,  3,  0, 31, 20,  0,  0,  1,  0,255,  0,  0,  0,255,255,  0,  0,255, // v,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  3,  0,  0,  3,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // w,
                                                                                                                          0,  0,  0,  1, 16,  0,  0, 22,  0,  0,  0,  3, 13,  0,  0,  0,  2,  2,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // x,
                                                                                                                          0,  0,  0,  0,  0,  0,  0, 57,  8,  0,  0,  1,  1, 61,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  6, 60,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // y,
                                                                                                                          0,  0,  0,  0,  1,  0,  0,  0, 14,  0,  0,  7,  3,  0,  6,  0,  3,  5,  0,  0,  0,  0,  4,  0,  1,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255, // z,
          0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // š,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // œ,
          0,  0, 29,  0,  0,  0, 15,  0,  0,  0, 10,  0,  0,  0,  0,  0, 20,  0,  0,  0,  0,  0, 36,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,255,  0,  0,  0,  0,255,  0,  0,255,255,  0,  0,  0,  0,  8,  0,  0,  0,  0,  2,255,  0,  0,  0,255,255,  0,255,255, // ß,
          0,  0,  0, 21, 16, 18, 14,  6, 23, 45, 13,  2,  0, 42, 17, 16,  0, 35,  0, 32,  4,254,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // à,
          0,  0,  0, 10,  8, 21, 34,  5,  5, 60, 17,  5,  1, 29, 40, 26,  2, 16,  0, 25,  9, 43, 28,  6,  0,  0,  1,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // á,
          0,  0,  0,  1,  2,  0,  0,  0,  0,  6,  0,  0,  0,  3,  1,  0,  0,  0,  0,  2,  0,  5,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // â,
          0,  0,  0,  0,  1,  2,  3,  0,  1,  2, 12,  0,  0,  1,  6, 28,  4,  1,  0, 11, 64, 10,  0,  1,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,124,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ã,
          0,  0,  1, 11,  0, 25,  6, 15, 11, 60, 24,122, 95,113, 68, 53,  1, 49,  0, 59, 97,197,  0, 75, 28,  0,  6, 12,  0,255,  0,  0,  0,  0,  0,118,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  2,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ä,
          0,  0,  0,  9, 14, 13,  2,  9, 34, 17,  3,  9,  4, 22, 41, 16,  0,149,  0,134, 53, 15,  0, 21,  8,  0,  6,  1,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,255,255,  0,255,255, // å,
          0,  0,  0,  6,  0,  3,  0, 19,  6, 12,  0, 15, 16, 46, 19, 19,  0,  7,  0, 67, 31, 19,  0, 58,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // æ,
          0,  0,109,  0,  2,  0, 15,  0,  0,  0, 34,  0,  0,  2,  0, 57,  9,  1,  0, 34,  0,  0,  8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ç,
          0,  0,  0,  9, 13, 19,  0, 15,  6, 70,138, 14, 45, 47, 90, 16,  3,122,  0, 36, 22,115, 50, 22, 42,  2, 79, 19,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 10,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,255,  0,  0,  0,255,255,  0,255,255, // è,
          0,  0,  2, 18, 23, 82,  0, 28, 22, 25, 24,  2,  8, 42, 50, 61,  1, 31,  0,119, 44,164, 14, 13, 29,  7,  4,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // é,
          0,  0,  0,  0,  3,  7,  0,  0, 10,  2,  3,  0, 25,  6,  5, 20,  0,  2,  0, 20,  1, 11,  4,  5,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ê,
          0,  0,  2, 23, 15, 36,140,  5, 23, 52, 50, 66, 48, 90, 56,215, 10,125, 35, 88, 58,254,  9, 24, 13,  0,  0,  8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ë,
          0,  0,  0,  0,  3,  6,  0,  3, 21,  9,  0,  0,  0, 17,  8,  3,  0,  1,  0, 64, 31,  8,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,  0,255,255,255,  0,255,255, // ì,
          0,  0, 71,  7, 10, 15,  2, 22,  9, 34,  0,  0,  0, 33, 14, 20,  1,  3,  0,195, 28, 55, 22, 10,  0, 10,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,255,  0,  0,  0,255,255,  0,255,255, // í,
          0,  0,  5,  1,  9,  5,  0,  0,  0, 21,  0,  9,  8,  7,  6,  9,  1, 10,  0, 19,  6,179,  0, 13,  0,  0, 23,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,255,  0,  0,255,255,  0,255,255, // î,
          0,  0,  6,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  9,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ï,
          0,  0,254,  0,  0,  0, 25,  0,  0,  0, 59,  0,  0,  0,  0,  0, 14,  0,  0,  0,  0,  0, 21,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,255,  0,255,255, // ñ,
          0,  0,  0, 56, 41,  8,159, 13,  3, 23, 17,  2,117, 31, 26, 45,  0, 20,  0, 20,  5, 23, 18,  6, 21,  4, 27, 63,  0,255,255,  0,  0,  0,  0,  0,  0,255,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,  0,255,255, // ò,
          0,  0,  0, 11, 14,  8,  7,  8, 66, 13,252,  3, 22, 13, 16, 16,  0,  7,  0, 29, 11, 25,  0,  5,  4,  1, 10, 12,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,255,255,  0,255,255, // ó,
          0,  0,  3,  3, 11,  1,  0,  2,  0,  7,  0,  1,  0,  2,  1,  7,  0,  6,  0,  6,  7,  4,  0,  2,  0,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ô,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0, 36,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,255,255,255,255,255, // õ,
          0,  0,  0, 32,  0, 24,  0,139,  5, 45, 10, 25, 31, 26, 13,  6,  0,  3,  0, 29,  8, 34,  0, 24,  5,  0, 44,  7,  0,255,  0,  0,  0,  0,255, 71,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  3,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ö,
          0,  0,  0, 17,  0, 43,  0, 90, 10, 34,  2, 44, 27, 34, 12, 10,  0,  2,  0, 39, 62, 38,  1,  8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,255,255,  0,255,255, // ø,
          0,  0,  0,  3,  3, 52,  0,  0,  2, 11, 64,  0,  0, 29,  0,  0,254,  0,  0,  6,  2,  3,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ù,
          0,  1,  2,  1,  7,  9,  1, 11,  5,  8, 41,  1,  0, 10,  7,  8,  0,  8,  0, 12, 27,  8,  0,  0,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,255,255,255,  0,255,  0,255,255,  0,255,255, // ú,
          0,  0,  0,  0,  0,  1,  5,  0,  0,  1,  0,  0,  0,  0,  0,  0, 44,  0,  0,  3,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,  0,  0,  0,  0,255,255,  0,  0,255,255,  0,255,255, // û,
          0,  0,  0, 18,  0,  6,  2,119,  9, 14,  0,  5, 15, 14, 26,  7,  0,  0,  2, 66, 16, 20,  0,  0, 16,  0,  0,  5,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ü,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,  0,255,255, // þ,
          0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ž,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ÿ,
          0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  2,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ·,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,  0,  0,255, // ¨,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0, // ®,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,255,  0,255, // ©,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,255,255, // ¬,
          0,  2,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,  0,255,255, // ª,
      //   ,  0,  a,  b,  c,  d,  e,  f,  g,  h,  i,  j,  k,  l,  m,  n,  o,  p,  q,  r,  s,  t,  u,  v,  w,  x,  y,  z,  š,  œ,  ß,  à,  á,  â,  ã,  ä,  å,  æ,  ç,  è,  é,  ê,  ë,  ì,  í,  î,  ï,  ñ,  ò,  ó,  ô,  õ,  ö,  ø,  ù,  ú,  û,  ü,  þ,  ž,  ÿ,  ·,  ¨,  ®,  ©,  ¬,  ª,
    ],
    icelandic: [
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, //  ,
                                                                                                                          0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0, // 0,
                                                                                                                          0,  2,  0,  0,  0,  6, 51,  2,  0,  0,  5,  0, 19,  0,255,255,  0,  0, // a,
                                                                                                                          0,  1,  0,  0,  0,  4,  2,  2,  0,  0,  0,  0,  0,  0,255,255,  0,255, // b,
                                                                                                                          0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // c,
                                                                                                                          0,  0,  0,  2,  0,  2,  0,  0,  0,  5,  0,  1,  0,  0,255,255,  0,  0, // d,
                                                                                                                          0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, 22,  0,255,255,  0,255, // e,
                                                                                                                          0,  4,  0,  1,  0,  5,  3,  3, 13,  0,  0,  0,  0,  0,255,255,  0,255, // f,
                                                                                                                          0,  4,  0,  3,  1,  7,  2, 10, 12, 19,  7,  3,  0,  0,255,255,  0,  0, // g,
                                                                                                                          0,  2,  0,  0,  0,  0,  3,  1,  0,  0,  0,  0,  0,  0,255,255,  0,255, // h,
                                                                                                                          0,  1,  0,  0,  0,  0, 68,  0,  0,  0,  1,  2,  4,  0,255,255,  0,  0, // i,
                                                                                                                          0,  0,  0,  1,  0,  0,  2,  0,  0,  0,  0,  1,  8,  0,255,255,  0,255, // j,
                                                                                                                          0,  3,  0,  5,  1, 21,  1, 10,  4, 10, 11,  0,  0,  0,255,255,  0,255, // k,
                                                                                                                          0, 30,  0,  8,  9,  4,  6, 78, 20, 18,  4,  1,  0,  0,255,255,  0,  0, // l,
                                                                                                                          0,  2,  0,  5,  0,  8,  2,  9,  1,  3,  1,  1,  0,  0,255,255,  0,  0, // m,
                                                                                                                          0,  9,  4,  4,  0, 11,  2, 18, 11,  6, 13,  3,  0,  0,255,255,  0,  0, // n,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,255,255,  0,255, // o,
                                                                                                                          0,  1,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // p,
                                                                                                                          0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0, // q,
                                                                                                                          0, 45,  1, 13,  7,  2,  7, 25, 17, 58,  9,  8,  7,  0,255,255,  0,  0, // r,
                                                                                                                          0,  8,  1,  2,  1, 37, 13,  5,  0,  1,  9,  9,  0,  0,255,255,  0,  0, // s,
                                                                                                                          0, 17,  0, 14,  7,  6,  1, 16,  3,  3, 14,  4,  0,  0,255,255,  0,  0, // t,
                                                                                                                          0,  0,  0,  0,  0,  7, 61,  0,  0,  0,  0,  3,  1,  0,255,255,  0,255, // u,
                                                                                                                          0,  5,  0,  2,  0,  3,  4,  3,  0,  9,  0,  0,  6,  0,255,255,  0,  0, // v,
                                                                                                                          0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // w,
                                                                                                                          0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,255,255,255,  0, // x,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,255,255,255,  0, // y,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255, // z,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,255,255,255,255,255, // š,
          0,  0,  0,  2,  0,  4,  0,  2,  3,  6,  0, 15,  3,  9, 23,  5,  0,  2,  0, 24,  3,  7,  0,  5,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  8,  0,255,255,255,255, // á,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,255,  0,  0,  0,  0,  1,  0,255,  0,  0,  0,255,  0,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255, // ä,
          0,  0,  0,  7,  0,  3,  0,  8,  1,  4,  0,  0,  0,  5,  2,  3,  0,  4,  0, 18,  5,  8,  0,  8,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  2,  0,255,255,255,255, // æ,
          0,  0,  0,  0,  0,  0,  0,  8,  0,  3,  0,  0,  0,  2,  0,  0,  0,  0,  0,  6,  6,  0,  0,  1,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255,255,255, // é,
          0,  0,  2,  2,  0,  1,  0,  1,  2,  0,  0,  0,  3, 16,  2,  5,  0,  1,  0, 25, 28, 19,  0, 29,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255, // í,
          0,  0, 95,  0,  0,  0, 24,  5,  5,  0,122,  0,  0,  0,  0,  0,  7,  0,  0, 46,  0,  0, 20,  2,  0,  0,  6,  0,  0,  8,255, 23,  0, 36,  0, 13,  7, 21,  1,  2,  0,  0,255,255,255,255, // ð,
          0,255,  0, 62,  0,  7,  0, 22,  2,  3,  4, 39,  9,  5,  4,  5,  0,  1,  0, 13,  8, 21,  0,  5,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  5,  0,255,255,255,255, // ó,
          0,  0,  0,  1,  0,  0,  0,  1,  2,  8,  0, 18,  2, 10,  4,  3,  0,  0,  0,  4,  6, 11,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,255,255, // ö,
          0,  0,  0,  3,  0,  3,  0, 40,  1, 11,  0, 14,  1, 13,  2,  2,  0,  0,255, 14,  9, 12,  0, 12,  0,  0,  0,255,  0,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255, // ø,
          0,  0,  0,  9,  0,  0,  0,  0,  2,  7,  0,  6,  3,  1,  1, 10,  0,  0,  0, 12,  3,  3,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  0,255,255,255,255, // ú,
          0,  0,  0,  9,  0,  2,  0,  0,  0,  0,  0,  0,  1,  6,  0,  7,  0,  0,  0,  1,  4,  9,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  4,  0,255,255,255,255, // ý,
          0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  2,  0,  0,  0,255,  0,  0,  0,  0,255,255,255,255, // þ,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255, // ·,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255, // ¨,
          0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255, // ®,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ©,
          0,  0,  0,255,  0,  0,255,255,  0,  0,255,255,  0,255,  0,  0,255,255,255,  0,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¬,
      //   ,  0,  a,  b,  c,  d,  e,  f,  g,  h,  i,  j,  k,  l,  m,  n,  o,  p,  q,  r,  s,  t,  u,  v,  w,  x,  y,  z,  š,  á,  ä,  æ,  é,  í,  ð,  ó,  ö,  ø,  ú,  ý,  þ,  ·,  ¨,  ®,  ©,  ¬,
    ],
    greek: [
                      0,  0,  0,  0,  0, 12,  0, 15,  6,  0, 68,  1,  0,  0, 24,  0, 33,  0, 40,  2,  2,  1, 50,  0, 44,  1,  2,104,  1,  2, 32,  0,  0,  0,  1,  0,  0, 15,  5,  1, //  ,
                      0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0, // 0,
                      0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // a,
          0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0, // ·,
          0,  0,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¨,
          0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ®,
          0,  0,255,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ©,
          0,255,  0,  0,255,255,255,255,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,255,  0,  0,  0,255,255,255,  0,  0,255,  0,255,  0, // ΐ,
          4,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,  1,  1,  0,  0,  0,  0,  0,  4,  7,  4,  4,  3,  0,  0,  3,  6,255,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ά,
          9,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  0,  0,  0,  0,  2,  1,  3,  2,  6,  2,  0,  0,  2,  2,255,  0,  5,  0,  1,  1,  0,  0,  0,  0,  0,  0,255, // έ,
          3,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,255,  0,  0,  1,  1,  0,  0,  0,  1,  0, 10,  3,  1,  2,  0,  0,  0,  2,255,  0,  4,  0,  1,  1,  0,  0,  0,255,  0,  0,  0, // ή,
          0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,255,  5,  1,  2,  2, 15,  0,  0,  0,  0,  1,  4,  1,  5,  0,  5,  2, 14,255,  3,  4,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0, // ί,
          0,255,  0,255,255,255,255,  0,255,  0,255,255,255,  0,255,255,  0,  0,255,  0,  0,  0,255,255,  0,255,255,  0,255,255,255,  0,255,  0,255,255,255,  0,255,  0,255,255,255, // ΰ,
         46,  0,  0,  0,255,255,  0,  0,  0,  1,  0, 22,  0,  0,  4,  3,  4,  0,  1,  0,  2, 20, 34,  8, 18, 23,  1,  0, 13, 18,255,  5, 30,  1,  4,  3,  0,  0,  0,  0,  0,  0,  0, // α,
         10,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // β,
         15,  0,  0,  0,255,255,  0,  0,  1,  1,  0,  0,  0,  7,  0,  1,  0,  2,  0,  6,  0,  1,  0,  0,  0,  0,  0,  5,  0,  5,255,  0,  0,  2,  0,  0,  0,  1,  0,  0,  1,  0,  0, // γ,
         18,  0,  0,  0,255,255,  0,  0,  3,  0,  0,  4,  0,  2,  0,  0,  0,  1,  0,  0,  0,  3,  0,  0,  0,  3,  0,  4,  0,  0,255,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // δ,
         33,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  2,255,  0,  2,  4,  7,  0,  2,  0,  5,  2,  9,  9, 18,  8,  1,  0,  9,  7,255, 14, 15,  0,  1,  5,  0,  0,  0,  0,  0,  0,  0, // ε,
          1,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ζ,
          9,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  3,  0,  0,  0,  4,  0,  2,  8,  3,  6,  1,  0,  1,  3,255, 14, 41,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0, // η,
          5,  0,  0,  0,255,255,  0,  0,  1,  0,  1,  0,255,  5,  0,  0,  0,  2,  0,  1,  0,  1,  0,  0,  0,  1,  0,  1,  0,  1,255,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // θ,
          6,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255, 28,  2,  8, 12, 19,  0,  0,  0,  0,  3,  9,  6, 10,  0, 12,  6, 19,255,  6, 19,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0, // ι,
         43,  0,  0,  0,255,255,  0,  0,  1,  1,  0,  2,  0,  8,  0,  5,  0,  5,  0,  4,  0, 33,  0,  0,  0,  0,  0,  3,  0,  1,255,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // κ,
          7,  0,  0,  0,255,255,  0,  0,  5,  3,  0,  1,  0, 10,  1,  1,  0,  9,  0,  2,  1,  3,  3,  9,  0,  0,  0, 10,  4,  0,255,  0,  0,  3,  0,  0,  0,  0,  0,  0,  3,  1,  0, // λ,
         30,  0,  0,  0,255,255,  0,  0,  1,  1,  3,  0,  0,  5,  0,  1,  0,  3,  0,  8,  1,  3,  0,  0,  2,  0,  0,  8,  0,  2,255,  5,  0,  4,  0,  0,  0,  2,  0,  0,  3,  1,  0, // μ,
         12,  0,  0,  0,255,255,  0,  0,  6,  8,  1,  7,  0, 29,  0,  1,  0, 10,  0, 14,  1,  7,  0,  0,  0,  1,  0, 16,  0,  1,255,  0,  0,  8,  0,  1,  0, 13,  0,  0,  5,  3,  9, // ν,
          1,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  1,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ξ,
         19,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  9,255,  0,  3,  9,  6,  0,  1,  0,  2, 14, 12, 13, 11, 14,  0,  0, 26, 21,255,  4, 53,  0,  4,  3,  0,  0,  0,  0,  0,  0,  0, // ο,
         44,  0,  0,  0,255,255,  0,  0,  1,  0,  0,  0,  0, 13,  0,  0,  0,  8,  0,  0,  0,  1,  0,  0,  7,  0,  0,  6,  0,  0,255,  2,  0,  5,  0,  0,  0,  1,  0,  0,  2,  1,  0, // π,
          3,  0,  0,  0,255,255,  0,  0,  5,  5,  1,  2,  0, 17,  3,  4,  4, 18,  0,  3,  1,  4,  5,  0,  0,  0,  0, 14, 12,  0,255,  0,  9,  5,  1,  4,  0,  2,  0,  0,  3,  2,  1, // ρ,
          0,  0,  0,255,255,255,  0,  0,  1,  6,  6,  0,  0, 14,  0,  0,  0,  9,  0, 21,  0,  6,  0,  0,  0,  0,  0, 17,  0,  0,255,  0,  0,  5,  0,  0,  0,  4,  0,  0,  7,  1,  0, // ς,
         43,  0,  0,  0,255,255,  0,  0,  3,  2,  3,  4,  0, 10,  0,  0,  0,  4,  0,  6,  0, 13,  0,  0,  0,  1,  0,  7,  0,  0,255,  2,  2,  4,  0,  0,  0,  3,  0,  0,  3,  2,  1, // σ,
         73,  0,  0,  0,255,255,  0,  0,  4,  2,  3,  3,  0, 21,  0,  0,  0, 10,  0,  5,  0,  4,  4,  0,  0, 16,  0,  3,  2,  2,255, 35,  0,  5,  0,  0,  0,  3,  0,  0,  7,  1,  1, // τ,
          4,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  4,  0,  0,  1,  5,  0,  0,  1,  0,  2,  1,  0,  0,  0, 53,  0,  1,255,  7,  1,  0,  1,  0,  0,  0,255,  0,  0,  0,  0, // υ,
          6,  0,  0,  0,255,255,  0,  0,  1,  0,  0,  0,  0,  5,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,255,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // φ,
          9,  0,  0,  0,255,255,  0,  0,  0,  3,  0,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  2,  0,  4,255,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // χ,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ψ,
          1,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  1,255,  0,  0,  1,  0,  1,  0,  0,  0,  1,  1,  1,  0,  3,  0,  0,  1,  4,255,  1,  6,  0,  1,  1,  0,  0,  0,255,  0,  0,  0, // ω,
          0,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,255,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0, // ϊ,
          0,255,  0,255,255,255,255,255,  0,  0,255,  0,255,  0,255,255,  0,  0,255,255,255,  0,  0,  0,255,  0,255,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0, // ϋ,
          6,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  1,  1,  0,  0,  0,  0,  2, 11,  1,  4,  3,  0,  0,  9,  5,255,  1,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ό,
          0,  0,  0,  0,255,255,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  1,  0,  0,  0, 12,  0,  0,255,  2,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255, // ύ,
          0,  0,  0,  0,255,255,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2,  1,  0,  2,  0,  0,  0,  3,255,  0,  1,  0,  0,  1,  0,  0,  0,255,  0,  0,  0, // ώ,
      //   ,  0,  a,  ·,  ¨,  ®,  ©,  ΐ,  ά,  έ,  ή,  ί,  ΰ,  α,  β,  γ,  δ,  ε,  ζ,  η,  θ,  ι,  κ,  λ,  μ,  ν,  ξ,  ο,  π,  ρ,  ς,  σ,  τ,  υ,  φ,  χ,  ψ,  ω,  ϊ,  ϋ,  ό,  ύ,  ώ,
    ],
    turkish: [
                                                                                                                      0,  0,  0,139,  7,218,164,  1,  1, 58, 17,238,138, 19,  1,  0,  0,  0,  0, //  ,
                                                                                                                      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                                                                                                                      0,  0,  0,  2, 15,  0,  0,  4,  0,  0,  1, 16,  0, 21,  0,255,255,  0,  0, // a,
                                                                                                                      0,  0,  0, 12,  0,  3,  5,  0,  0,  1,  1, 53,  2,  0,  0,255,255,  0,  0, // b,
                                                                                                                      0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  1, 22,  9,  0,  0,255,255,  0,  0, // c,
                                                                                                                      0,  0,  0, 22,  0, 11,  4,  0,  2,  1,  2,160,  5, 16,  0,255,255,  0,  0, // d,
                                                                                                                      0,  0,  0,  0, 16,  0,  0,  4,  0,  1,  0, 12,  0, 13,  0,255,255,  0,  0, // e,
                                                                                                                      0,  0,  0, 12,  0,  0,  1,  0,  0,  0,  6, 19,  2,  1,  0,255,255,  0,  0, // f,
                                                                                                                      0,  0,  0,  0,  0,  1,  2,  0,  0,  2,  0, 33,  0,  0,  0,255,255,  0,  0, // g,
                                                                                                                      0,  0,  0, 21,  0,  1,  2,  0,  0,  0,  2, 20,  0,  0,  0,255,255,  0,  0, // h,
                                                                                                                      0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  7,  0,  0,  0,255,255,  0,  0, // j,
                                                                                                                      0,  0,  0, 30,  0,  8, 25,  0,  1,  4, 10, 82, 27,  4,  0,255,255,  0,  0, // k,
                                                                                                                      0,  0,  0, 45,  3,  3,  9,  9, 14,  1, 15,113, 43, 13,  0,255,255,  0,  0, // l,
                                                                                                                      0,  0,  0, 19,  1,  6,  5,  0,  1,  2, 16, 76, 28,  7,  0,255,255,  0,  0, // m,
                                                                                                                      0,  0,  0, 63,  0, 88, 46,  0, 20, 13, 26,150, 76,  0,  0,255,255,  0,  0, // n,
                                                                                                                      0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0, 54,  1,  1,  0,255,255,  0,  0, // o,
                                                                                                                      0,  0,  0,  0,  0,  0,  3,  0,  0,  3,  4, 16,  1,  0,  0,255,255,  0,  0, // p,
                                                                                                                      0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  1, 12,  8,  1,  0,255,255,  0,  0, // q,
                                                                                                                      0,  0,  0,123,  0, 21, 23,  4, 16, 10, 30,101, 32,  1,  0,255,255,  0,  0, // r,
                                                                                                                      0,  0,  0, 47,  0,  0, 17,  0,  1,  3,  7, 88, 12,  0,  0,255,255,  0,  0, // s,
                                                                                                                      0,  0,  0, 35,  1,  2,  7,  0,  0,  1,  4, 89,  4, 18,  0,255,255,  0,  0, // t,
                                                                                                                      0,  0,  0,  0,  1,  0,  0, 16,  0,  0,  0,  2,  0,  5,  0,255,255,  0,  0, // u,
                                                                                                                      0,  0,  0,  4,  0,  2,  2,  0, 11,  0,  1, 22,  1,  0,  0,255,255,  0,  0, // v,
                                                                                                                      0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  2,  0,  0,  0,255,255,  0,  0, // w,
                                                                                                                      0,  0,255,  3,  0,  3,  0,  0,  0,  0,  1,  9,  3,  0,  0,255,255,  0,  0, // x,
                                                                                                                      0,  0,  0, 14,  0,  1,  5,  0,  6,  5,  5, 72,  6,  0,  0,255,255,  0,  0, // y,
                                                                                                                      0,  0,  0, 10,  0, 10,  5,  0,  7,  0, 19, 24,  8,  0,  0,255,255,  0,  0, // z,
          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255, // š,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255, // œ,
          0,  0,  0,255,  0,  0,  0,255,255,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,255,255,  0,255,255,  0,  0,  0,  0,255,  0,255,  0,255,  0,  0,  0,255,255,255,255,255,255, // ß,
         36,  0,  0, 20, 12, 81,  0, 14,  9, 24,  0, 17,101, 55, 24,  0,  1, 12, 35, 31, 42,  0, 36,  0,  7, 21, 21,  0,  0,  0,  0,  4,255,  0,  0,  0,255,  3,  1,  0, 24,  0,255,255,  0,255, // ä,
         29,  0, 12,  0,  0,  0,  8,  0,  0,  1,  0,  7,  5,  1,  1,  1,  0,  0,  2,  0,  2,  2,  7,  0,  1,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  5, 13,  0,  1,  0,255,255,  0,255, // ç,
         22,  0,  0,  9,  1, 20,  0,  1,  6, 13,  7, 17, 18, 14, 47,  0, 12,  0, 57,  7, 23,  0, 20, 23,  2, 57,  5,  0,  0,  0,255,  3,  0,  0,  0,  0,  0,255,  0,  0,  6,255,255,255,  0,255, // ê,
         19,  0,  0,  6,  5, 24,  0,  4,  2,  6, 16, 25, 21, 14, 33,  0,  4,  1, 51, 12, 20,  0, 13, 17,  0, 16, 10,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  7,255,255,255,  0,255, // î,
          0,  0, 25,  0,  0,  0,  9,  0,  0,  0,  0,  0,  0,  0,  0, 13,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1, 11,  9,  0,  0,255,255,  0,255, // ğ,
         22,  0,  0,  8,  0,  6,  0,  0, 16,  1,  0,  7,  0,  2,  4,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0, 10,  1,255,  0,255,  0,  1,255,255,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,255, // ö,
         45,  0,  0, 20,  0,  4,  0,  0,  0,  1,  0,  2,  1,  4,  5,  0,  1,  0, 10,  2,  5,  0,  0,  0,  0,  1,  0,  0,  0,255,255,  3,  0,  0,  0,255,  0,  0,  0,  0,  3,255,255,255,255,255, // û,
         14,  0,  0,  8,  5, 18,  0,  0, 11,  3,  0,  8, 19, 21, 12,  0,  0,  1, 10,  6, 24,  0,  2,  0,  1, 12,  2,  0,  0,  0,  0,  6,255,  0,  1,  0,  0,  0,  0,  0,  2,  0,255,255,  0,255, // ü,
        161,  0, 11, 73, 23,124,  6, 22, 26, 38, 38, 87,117, 51, 60,  6, 16,  7,182,108,173,  3, 49, 61,  9, 15, 18,  0,  0,  0,  0, 19,  0,  0, 16,  0,  0,  0,  1,  0, 25,  0,255,255,  0,  0, // i,
         18,  0,  0,  1, 16, 39,  0,  6,  2,  4,  0, 27, 49, 19, 40,  0,  7,  7, 48, 46, 27,  0,  0,  0,  4, 31,  7,  0,  0,  0,  0, 11,  0,  0, 20,  0,255,  0,  0,  3, 12,  0,255,255,  0,255, // ı,
         32,  0, 32,  1,  0,  0, 15,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0, 10,  0,  0,  8,  0,  1,  5,  0,  0,255,  0,  0, 11,  0,  6,  5,  0,  0,  2,  4, 24, 20,  0,  0,255,255,  0,255, // ş,
          1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255, // ·,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0, // ¨,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0, // ®,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,255,255, // ©,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,  0,255,  0,255, // ¬,
      //   ,  0,  a,  b,  c,  d,  e,  f,  g,  h,  j,  k,  l,  m,  n,  o,  p,  q,  r,  s,  t,  u,  v,  w,  x,  y,  z,  š,  œ,  ß,  ä,  ç,  ê,  î,  ğ,  ö,  û,  ü,  i,  ı,  ş,  ·,  ¨,  ®,  ©,  ¬,
    ],
    hebrew: [
                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 27, 28, 16, 21,124, 42, 25,  7, 82, 64, 20,  2, 68, 95,  2,191,  0, 33, 57, 10,  5,  6,  2, 12, 97, 13,127, //  ,
                          0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,  0,  0,255,255,255,  0,255,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                          0,255,255,  0,  0,255,255,  0,255,  0,255,  0,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // a,
                          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  1,  1,  1, // .,
          0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0, // ·,
          0,  0,255,  0,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¨,
          0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,  0,255,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,255,  0,255,  0,255,255,255,255,  0,255,255, // ®,
          0,  0,255,  0,255,255,  0,255,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ©,
          0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¬,
          0,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ₪,
          0,  0,255,  0,  0,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ‎,
          0,  0,255,  0,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0, // ְ,
          0,255,255,  0,255,255,255,255,255,255,255,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ֱ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ֲ,
          0,255,255,  0,255,255,255,255,255,255,255,255,255,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ֳ,
          0,255,  0,  0,255,255,255,255,255,255,255,  0,255,255,  0,  0,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ִ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,255,255,  0,  0,255,255,255,255,255,255,255,255,255,255,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0, // ֵ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,  0,255,255,  0,  0,255,255,255,255,255,255,255,255,255,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ֶ,
          0,255,  0,  0,255,255,255,255,255,255,255,  0,255,  0,255,  0,  0,  0,  0,255,255,255,255,255,255,255,255,  0,255,  1,255,255, 20,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0, // ַ,
          0,255,  0,  0,255,255,255,255,255,255,255,  0,255,255,255,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,255,255,255, 14,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ָ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0, // ֹ,
        255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,255,255,  0,  0,255,  0,255,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,  0,  0,255,  0, // ֺ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,  0,255,255,255,255,  0,255,  0,255,  0,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ֻ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0, // ּ,
          0,255,255,255,255,255,255,255,255,255,255,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,255,255,255,255,255,255,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,255,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0, // ֽ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0, // ׁ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0, // ׂ,
          1,255,  0,  0,255,255,255,255,255,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0, // װ,
          0,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,  0,255,255,255,  0,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,255,  0,255,  0,  0,255,255,  0,255,  0,  0,  0,  0,255, // ױ,
          0,255,255,  0,255,255,255,  0,255,255,255,  0,255,255,255,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ײ,
          0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,255,255,  0,255,  0,  0,  0,255,255,255,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ׳,
          0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,255,255,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ״,
        168,  0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 20, 14, 15, 21, 22,  5,  0, 22, 15,  0,  2, 17,  0, 16,  0, 16,  6,  2,  0, 27,  0,  3, 25, 30,  5,  3, // א,
         99,  0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0, 10,  2,  2,  2,  7, 22,  0,  4,  1, 15,  0,  3,  6,  0,  5,  0,  2,  1,  7,  0,  0,  0,  3,  5, 15,  6,  4, // ב,
         43,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  1,  0,  2,  5,  9,  0,  0,  6, 12,  0,  0,  3,  0,  2,  0, 17,  1, 15,  0,  1,  0,  0,  0,  7,  1,  1, // ג,
         68,  0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  8,  4,  4,  0,  4, 16,  0,  5,  0, 15,  0,  3,  5,  0,  8,  0, 20,  3,  7,  0,  2,255,  1,  2,  4,  0,  0, // ד,
        155,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  8,  2,  6,  3,  8,  5,  4,  2, 36,  0,  4, 17,  0, 11,  0, 16,  1,  4,  0,  4,  0,  2,  5, 16,  9,  5, // ה,
         42,  0,  0,  0,  0,255,255,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0, 47, 21, 16, 18, 19, 73,  5, 13, 12, 38,  0, 11, 22,  0, 30,  0, 25, 10, 13,  0, 32,  0, 18, 22, 34, 19, 17, // ו,
         31,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  8,  0,  0,  2,  1,  6,  0,  2,  0, 25,  0,  1,  0,  0,  2,  0,  1,  0,  4,  0,  0,255,  0,  0,  0,  0,  0, // ז,
         17,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  8,  4,  0,  0,  6,  7,  0,  0,  1, 10,  0,  1,  5,  0,  6,  0,  2,  0,  0,  0,  1,  0,  1,  0,  4,  4,  5, // ח,
         19,  1,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  7,  1,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0, 28,  1,  4,  0,  2,  6,  0,  0,  0, 30,  0,  5,  9,  0,  4,  0, 20, 21, 11,  0,  7,  0,  1, 10, 16, 20,  0, // ט,
         30,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0, 65, 31,  9, 45, 19, 60, 22, 12, 19, 77,  0,  9, 37,  0, 25,  0, 45, 15, 12,  0, 15,  0, 15, 17, 50, 25, 13, // י,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  5,  0,  0,  0,  0,  2,  0,  0,  0, 10,  0,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  4,  0,  0, // ך,
         28,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  5,  2,  0,  0,  4,  7,  2,  0,  0,  5,  0,  0,  4,  0,  4,  0,  1,  1,  3,  0,  0,255,  0,  0,  6,  2,  1, // כ,
         62,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 35,  7,  8,  2,  5, 29,  1,  6,  4, 20,  0,  8,  4,  0,  7,  0,  1,  2, 18,  0,  4,  0,  2,  6,  2, 29,  2, // ל,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  2,  0,  3,  1,  3,  9,  0,  0,  0, 65,  0,  0,  2,  0,  0,  0,  0,  0, 11,255,  0,255,  0,  0,  0,  5,  1, // ם,
         86,  0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0, 12, 11,  1,  2, 21, 15,  2,  3,  1, 10,  0,  2,  8,  0,  5,  0,  1,  2,  7,  0,  0,  0,  2,  1,  5,  7, 10, // מ,
          1,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 13,  8,  3,  3,  1, 48,  1,  0, 12, 36,  0,  3,  3,  0,  3,  0,  0,  2, 31,  0,  3,  0,  1,  1, 15,  5,  1, // ן,
         23,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 35,  8,  2,  1,  5, 26,  0,  1,  1, 35,  0,  4,  4,  0,  8,  0,  0,  1, 30,  0,  5,  0,  0,  2,  3, 13,  2, // נ,
         20,  1,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16,  4,  0,  0,  6, 14,  0,  2,  1, 17,  0,  1,  2,  0,  6,  0,  5,  0, 12,  0,  2,  0,  0,  3,  5,  0,  0, // ס,
         33,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 44, 38, 14, 25,  8,  0, 46, 36,  0,  5, 20,  0, 25,  0, 25,  5,  1,  0, 11,  0,  5, 15, 19, 17,  1, // ע,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ף,
         74,  0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 13,  2,  0,  0,  8, 12,  0,  0,  0,  9,  0,  1,  4,  0,  5,  0,  2, 10,  7,  0,  0,  0,  1,  1,  4,  5,  3, // פ,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255,  0,  0,  0,  0,  2,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, // ץ,
         24,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  4,  1,  0,  0,  5,  7,  0,  0,  0,  7,  0,  0,  1,  0,  5,  0,  5,  0,  2,  0,  0,  0,  0,  2,  3,  0,  0, // צ,
         38,  0,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  3,  0,  1,  6,  9,  0,  4,  0, 17,  0,  0,  5,  0,  7,  0,  5,  3,  9,  0,  3,  0,  0,  0,  5,  1,  4, // ק,
         26,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  3,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 60, 25,  8,  9, 10, 49,  3,  9,  9, 22,  0,  3,  2,  0, 11,  0,  2,  4,117,  0, 18,  0,  5, 10,  0, 11,  9, // ר,
         74,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  8,  7,  0,  2, 10,  9,  4,  2,  7, 32,  0,  1,  4,  0, 26,  0,  3,  0,  5,  0,  2,  0,  0,  2,  6,  0,  2, // ש,
         20,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0, 17,  8,  1,  2,  9, 64,  0,  6,  1, 34,  0,  5,  5,  0,  8,  0,  8,  2,  4,  0,  5,  0,  1,  1, 10, 11,  1, // ת,
      //   ,  0,  a,  .,  ·,  ¨,  ®,  ©,  ¬,  ₪,  ‎,  ְ,  ֱ,  ֲ,  ֳ,  ִ,  ֵ,  ֶ,  ַ,  ָ,  ֹ,  ֺ,  ֻ,  ּ,  ֽ,  ׁ,  ׂ,  װ,  ױ,  ײ,  ׳,  ״,  א,  ב,  ג,  ד,  ה,  ו,  ז,  ח,  ט,  י,  ך,  כ,  ל,  ם,  מ,  ן,  נ,  ס,  ע,  ף,  פ,  ץ,  צ,  ק,  ר,  ש,  ת,
    ],
    arabic: [
                      0,  0,  4,  0,  0,  0,  0,  0,  7, 10,  1,  0,  8,  7, 36,  0, 30,  5, 66,  7,  1,  0,  0,  0,  9, 66, 18, 77, 36,  2,  3,  3,  5, 44,  0, 69, 20, 26,  8,  2,  2,  4,  5, 11,  6,  0, 19,  6,  4, 31, 37, 62, 76, 33,  9, 39,  1,  0,  0,  0,  0,  0,  0,  0, 44, //  ,
                      0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                      0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // a,
         10,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  3,  4,  8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ‌,
          0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,  0,  0,  0,  0,  0,255, // ‍,
          1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ؛,
          0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255,255,  0,255,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0, // ·,
          0,  0,255,255,255,  0,  0,255,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¨,
          0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ®,
          0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ©,
          0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ¬,
         17,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0, // پ,
          7,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  1,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0, // ٹ,
          3,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0, // چ,
          1,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0, // ژ,
          7,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  1,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0, // ڈ,
          8,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,255,  1,  0,  0,  0,  0,  3,  0,  2,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // گ,
         70,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  2,  0,  4,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  4,  7,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ک,
          0,255,  0,  0,  0,  0,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,  0,  0,  0,  0,  0, // ڑ,
         24,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  2,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0, // ں,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  3,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0, // ھ,
         24,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  3,  2,  0,  3,  0,  0,  0,  0,  5,  0, 25,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  3,  1,  0,  9,  1,  2,  0,  2,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0, // ہ,
          0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ء,
         13,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // آ,
         21,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  9,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // أ,
          0,  0,  0,  0,  0,255,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255, // ؤ,
         10,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,255,255,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // إ,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ئ,
        111,  0,  0,  5,  0,  0,  0,255,255,  0,255,  6,  5,  0,  0,  1,  7, 17,  0,  0,  3,  7,  0,  0,  0,  0,  0,  0,  0, 24,  0, 18,  2,  9,  5,  2, 13,  1, 23,  4, 11,  4,  2,  2,  2,  1,  7,  1,  0,  5,  7,  7, 19, 13, 14, 20, 17,  0, 15,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ا,
         44,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0, 11,  0,  0,  3,  0,  0,  0,  0,  0,  0,  9,  2,  4,  0,  0,  0,  1,  0,  3,  0,  0,  0,  1,  1,  6,  2,  1,  0,  7,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ب,
          0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,255,  0,  0,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  3,  0,  0,  0,  1,  1,  0,  4,  0, 10,  0,  1,  0,  0,  0,  1,  0,  3,  1,  0,  2,  3,  1,  8,  4,  4,  0,  0,  0, 21,  0,  0,  0,  0,  0,  0,  0,  0,255, // ة,
         34,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 20,  1,  0,  0,  0,  0,  1,  2,  0,  0,  3,  0, 24,  3,  0,  0,  0,  0,  1,  0,  0,  2,  1,  2,  9,  6,  5,  0,  5,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ت,
          1,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ث,
         19,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  1,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  1,  1,  0,  3,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ج,
          8,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2,  1,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  4,  3,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ح,
         10,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  1,  0,  2,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0, // خ,
         37,  0,  0,  1,  0,  0,  0,255,255,  0,255,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 17,  2,  0,  0,  0,  1,  4,  1,  2,  0, 27,  1,  1,  5,  1,  0,  0,  0,  4,  0,  0,  0,  4,  0,  5,  4, 15,  1,  8,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0, // د,
          1,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  1,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ذ,
         54,  0,  0,  0,  0,  0,  0,255,255,  0,255,  6,  2,  0,  0,  0,  5,  9,  0,  0,  0,  6,  0,  1,  2,  0,  0,  1, 31, 12,  0,  6,  0,  1,  1,  2, 18,  1,  4,  0,  2,  4,  1,  0,  1,  0,  3,  2,  0,  5,  2,  4,  3, 27,  1,  8, 14,  0,  8,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ر,
         27,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ز,
         27,  0,  0,  1,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0, 17,  1,  0,  1,  0,  0,  1,  0,  1,  0,  9,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  6,  9,  3,  1,  5,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0, // س,
         19,  0,  0,  2,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  2,  0,  0,  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2,  3,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ش,
          7,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0, 14,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ص,
          1,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ض,
          1,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  3,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  1,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ط,
          0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0, // ظ,
         16,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  5,  2,  0,  3,  0,  3,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  8,  6,  0,  0,  5,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ع,
          2,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  6,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // غ,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ـ,
         21,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  5,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  1,  0,  3,  0, 16,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ف,
          8,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  1,  0,  3,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  5,  2,  1,  0,  4,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ق,
         11,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  6,  1,  0,  0,  1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ك,
         15,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1,  1,  0,  3,  0,122,  6,  0,  1,  1,  1,  1,  2,  1,  0,  0,  0,  3,  0,  2,  1,  1,  0, 10,  0,  0,  1,  3,  2,  5,  8,  0,  2, 12,  0, 10,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ل,
         68,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  3,  3,  0,  0,  0, 16,  0,  0,  2,  0,  3,  2,  0,  4,  0,  8, 22,  6,  2,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1, 27,  1,  1,  4,  6,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0, // م,
         34,  0,  0,  1,  0,  0,  0,255,255,  0,255,  1,  0,  0,  0,  0,  1,  4,  0,  0,  0,  1,  0,  2,  3,  4,  3,  1, 55,  2,  0,  2,  0,  2,  0,  0,  2,  0,  2,  2,  3,  2, 13,  0,  1,  0,  3,  0,  0,  0,  0,  2,  4, 14,  1,  2, 10,  0, 11,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ن,
         14,  0,  0, 11,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 10, 13,  0,  4,  0,  1,  0,  0, 30,  0,  6,  2,  1,  5,  0,  0,  0,  0,  1,  0,  0,  2,  1,  0,  4,  3,  4,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ه,
         32,  0,  0,  0,  0,  0,  0,255,255,  0,255,  1,  0,  0,  0,  0,  2,  6,  0,  0,  1,  6,  0,  0,  4,  0,  0,  0, 10,  9,  0,  5,  0,  9,  4,  4,  5,  0, 13,  1,  3,  3,  3,  0,  1,  0,  1,  0,  0,  2,  1,  4,  9,  6, 10,  3,  1,  0,  8,  0,  0,  0,  0,  0,  0,  0,  0,  0, // و,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ى,
          8,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  1,  0,  1,  2,  5,  8,  0,  5,  0,  2,  2,  1, 12,  1, 21,  2,  8,  2,  1,  1,  1,  0,  2,  1,  0, 19,  3,  4, 14,  7, 23,  2, 10,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ي,
          0,  0,  0,  0,  0,  0,  0,255,255,255,255,  0,255,  0,255,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255, // ً,
          0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,  0,255,255,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255, // ٌ,
          0,  0,  0,  0,255,  0,255,255,255,255,255,  0,255,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,  0, // ٍ,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,  0, // َ,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255, // ُ,
          0,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0, // ِ,
          0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0, // ّ,
          0,  0,  0,  0,255,  0,255,255,255,255,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ْ,
          2,  0,  0,255,  0,  0,255,255,255,  0,255,  0,  0,  1,  0,  0,  0, 11,  0,  0,  0,  6,  0,  0,  0,  0,255,  5,  0,  0,255,  1,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  4,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,255,  0, // ے,
      //   ,  0,  a,  ‌,  ‍,  ؛,  ·,  ¨,  ®,  ©,  ¬,  پ,  ٹ,  چ,  ژ,  ڈ,  گ,  ک,  ڑ,  ں,  ھ,  ہ,  ء,  آ,  أ,  ؤ,  إ,  ئ,  ا,  ب,  ة,  ت,  ث,  ج,  ح,  خ,  د,  ذ,  ر,  ز,  س,  ش,  ص,  ض,  ط,  ظ,  ع,  غ,  ـ,  ف,  ق,  ك,  ل,  م,  ن,  ه,  و,  ى,  ي,  ً,  ٌ,  ٍ,  َ,  ُ,  ِ,  ّ,  ْ,  ے,
    ],
    baltic: [
                                                                                                                          0,  0,  0, 57, 42,135,  2, 14, 20,  3,  2,119,  0,  0, 18,  1, 18,  0,  0,  1,  4,205,  1,  3,  4,  0,  0,  0,  0,  0, //  ,
                                                                                                                          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // 0,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1,  0, 16, 38, 14,  0,  0,  0,  0,  0,  1, 16,  0,255,255,  0,  0, // a,
                                                                                                                          0,  0,  0,  0,  0,  3,  8,  0,  0,  0,  0,  0,  0,  0, 37,  0,  0,  0,  0,  3,  0,  0,  0,  2,  0,  0,255,255,  0,  0, // b,
                                                                                                                          0,  0,  0,  0,  0, 19,  0,  0,  9,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,255,255,  0,  0, // c,
                                                                                                                          0,  0,  0,  0,  0, 17,  4,  0,  6,  0,  0,  6,  0,  0, 14,  0,  0,  0,  0,  8,  6,  0,  6, 11,  3,  0,255,255,  0,  0, // d,
                                                                                                                          0,  0,  0,  0,  0,  0, 32,  0,  0,  9,  0,  0,  5,  3,  0,  1,  7,  3,  0, 12,  1,  0,  0,  0, 21,  0,255,255,  0,  0, // e,
                                                                                                                          0,255,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // f,
                                                                                                                          0,  0,  0,  0,  1,  1, 15,  0,  1,  0,  0,  3,  0,  0, 27,  0,  0,  0,  0,  7,  1,  0,  1,  4,  0,  0,255,255,  0,  0, // g,
                                                                                                                          0,  0,  0,  0,  0,  0, 29,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 28,  0,  0,  0, 44,  0,  0,255,255,  0,  0, // h,
                                                                                                                          0,  0,  0,  0,  0,  0, 49,  0,  0, 72,  0,  0, 13,  9,  0,  5, 41,  7,  0,111,  2,  0,  0,  0, 56,  0,255,255,  0,  0, // i,
                                                                                                                          0,255,  0,  5,  0, 30,  0,  0, 32,  0,  0, 41,  0,  0, 12,  0,  0,  0,  0, 16,  0,  3,  0,  0,  0,  0,255,255,  0,  0, // j,
                                                                                                                          0,  0,  0,  0,  4, 32,  4,  0,  8,  0,  0,  2,  0,  0,  3,  0, 36,  0,  0,  2,  3,  0,  6, 19,  1,  0,255,255,  0,  0, // k,
                                                                                                                          0,  0,  0,  0,  0, 29, 54,  0, 36,  1,  0, 24,  0,  0,  4,  0,  5,  0,  0, 24,  3,  0,  2, 76,  0,  0,255,255,  0,  0, // l,
                                                                                                                          0,  0,  0,  0,  1, 16,  2,  0, 11,  0,  0,  6,  0,  0, 15,  0,  2,  0,  0,  3,  1,  0,  1, 35,  6,  0,255,255,  0,  0, // m,
                                                                                                                          0,  0,  0,  0,  0, 19, 49,  0,  7,  0,  0, 10,  0,  0, 12,  0,  8,  0,  0, 20,  2,  0, 16, 31,  6,  0,255,255,  0,  0, // n,
                                                                                                                          0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  2, 10,  2,  0,  0,  0,  0,  0,  0,  6,  0,255,255,  0,255, // o,
                                                                                                                          0,  0,  0,  0,  0,  3,  5,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0, 28,  1,  0,  1,  7,  0,  0,255,255,  0,  0, // p,
                                                                                                                          0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // q,
                                                                                                                          0,  0,  0,  2,  2, 59,121,  0, 23,  0,  1,  2,  0,  0,  6,  0,  3,  0,  0, 28, 10,  0, 23, 17,  0,  0,255,255,  0,  0, // r,
                                                                                                                          0,  0,  0,  2,  7, 50, 11,  7,  9,  1,  0, 88,  0,  0,  7,  0,  4,  0,  0,  4,  4,  0,  5, 19,  1,  0,255,255,  0,  0, // s,
                                                                                                                          0,  0,  0,  0,  2, 33, 11,  0, 31,  0,  0, 10,  0,  0, 21,  0, 22,  0,  0, 22,  8,  0,  6, 12,  1,  0,255,255,  0,  0, // t,
                                                                                                                          0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1,  0, 13, 11, 12,  0, 36,  0,  0,  0,  0,  7,  0,255,255,  0,  0, // u,
                                                                                                                          0,255,  0,  0,  5, 10,  2,  0,  2,  0,  0,  3,  0,  0, 21,  0, 12,  0,  0,  1,  1,  0,  1,  1,  3,  0,255,255,  0,  0, // v,
                                                                                                                          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // w,
                                                                                                                          0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0, // x,
                                                                                                                          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  5,  0,  0,255,  0,  0,  0,  0,  7,  0,255,255,  0,  0, // y,
                                                                                                                          0,255,  0,  0,  0,  4,  0,  0,  2,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,255,255,  0,  0, // z,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,255,255,255,255,255,255,255,255,255,  0,255,255,  0,  0,255,255,  0,  0,  0,255,255,  0,255, // ø,
          0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,255,255,255,255,255,  0,255,255,255,  0,255,255,  0,255,255,255,255,255,  0,255,255,255,  0,  0,255,255,  0,255,255,255,255,255,255,255, // ŗ,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,255,  0,255,255,255,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255, // ß,
          1,  0,  0,  1,  0,  1,  0,  0,  1,  0,  4,  9,  4,  3, 10,  5,  0,  0,  0,  5,  9,  9,  0,  1,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,255,  0,255,  0,255,255,255,255,  0,255,255,255,255,  0,  0,255,  0,  0,255,255,255,255, // ą,
         50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  5,  0,  0,  0,  3,  1,  2,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,  0,255,  0,255,  0,255,255,255,255,  0,255,255,255,255,  0,  0,255,  0,  0,255,255,  0,255, // į,
          9,  0,  0, 10,  2, 28,  3,  0, 13,  1,  6, 44, 45, 27, 28, 50,  0, 30,  0, 40, 13, 73,  2, 28,  0,  0,  0,  5,255,  0,  0,255,255,  0,255,255,  0,  0,  0,255,  0,  0,  0,  2,  4,  1,  0,255,255,255,  0,255,  3,  0,255,255,  0,255, // ā,
          8,255,  0,  1,  0,  0,  0,  1,  0, 11,  0, 56, 19, 30, 64, 25,  0, 45,  0,  7,  5, 41,  0, 86,  0,  0,  0,  0,  0,255,  0,  0,255,255, 48,255,255,  0,255,255,255,255,255,255,  0,255,255,  0,  0,255,255,  0,  0,  0,255,255,  0,255, // ä,
          0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  2,  1,  0,  4,  0,  0,  0,  1,  0,  2,  0,  2,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,  0,255,  0,255,  0,255,255,255,255,  0,255,255,255,255,  0,  0,255,  0,255,255,255,255,255, // ę,
          4,255,  0,  3,  6, 12,  0,  0,  0,  0,  0,  2,  0, 20, 16,  8,  0, 35,255, 15, 19, 28,  0, 26,  0,  0,  0,  5,  0,  0,  0,255,255,  0,255,255,  0,  0,255,255,  0,  1,  0,  0,  1,  3,  0,  0,  0,255,  0,255,  0,  0,255,255,  0,255, // ē,
         12,  0,  9,  0,  0,  0,  9,  0,  0,  0, 15,  0,  0,  1,  0,  8,  2,  0,  0,  4,  7,  0,  3,  0,  0,  0,  6,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  7,  0,  0,  0,  0,  0,  1,  0,  0,  0,255,255,  0,255, // č,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,255,255,  0,  0,255,255,255,255,255,  0,255,255,  0,255,255,255,  0,  0,  0,255,255,  0,255, // é,
          1,  0,  0, 20,  0, 27,  0,  0,  9,  0,  0,  3, 27, 33, 22, 68,  0, 12,  0, 25, 12, 29,  0, 20,  0,  0,  0,  1,255,255,  0,  0,  0,255,255,  0,255,  0,255,  0,255,255,255,  0,  2,255,  0,255,255,  0,  0,255,  4,  0,255,255,255,255, // ė,
          6,  0,  1,  0,  0,  0,  2,255,  0,  0,  1,255,  0,  0,  0,  0,  3,  0,255,  1,  0,  0,  1,  0,255,255,255,  0,255,255,255,255,255,  0,255,255,  1,255,255,255,  0,255,  0,  0,  0,  0,255,255,255,255,  0,255,  0,  0,255,255,  0,255, // ģ,
          4,255,  1,  0,  0,  0,  1,255,  0,255,  3,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,255,255,255,  0,255,255,  0,  0,255,255,  0,  0,  0,  0,  5,  2,255,255,255,255,  0,255,  0,  0,255,255,  0,255, // ķ,
          4,  0,  0,  4, 11, 17,  0,  0,  0,  0,  0,  0,  0, 32,  4, 17,  0,  2,  0, 44,  6, 35,  0,  7,  0,  0,  0, 25,255,  0,  0,255,255,  0,255,255,  0,  0,255,255,  0,  2,  0,  0,  3,  0,  0,255,255,255,  0,255,  0,  0,255,255,  0,255, // ī,
          2,255, 11,  0,  0,  0, 10,  0,  2,  0,  3,  0, 11,  0,  0,  0,  2,  0,255,  0,  0,  0,  2,  0,255,255,255,  0,255,  0,255,255,255,  1,255,255,  2,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,255,255,  0,255, // ļ,
         76,  0, 31,  0,  0,  0, 15,  0,  1,  0, 71,  0, 18,  1,  1,  1, 13,  2,  0,  7,  0, 10, 10,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2,  0,  0,  4,  0,  0,  0,  0,  0,  3,  1,  0,  5,  0,  0,  0,  0,  3,  0,  0,  0,255,255,  0,255, // š,
          1,255,  5,  0,  0,  0,  6,  0,  0,  0, 23,  0,  0,  0,  0,  0,  2,  0,255,  1,  0,  0,  1,  0,255,255,255,  2,255,  0,  0,255,255,  3,255,255,  0,  0,255,255,255,  0,  1,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,255,255,  0,255, // ņ,
          0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,255,  0,  0,  0,255,255,255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,255,  0,255, // ō,
         29,  0,  2,  0,  0,  0,  0,  0,  0,  5,  2, 22, 29, 25, 38, 18,  0, 32,  0,  4, 38, 23,  0, 87,  0,  0,  0,  0,  0,  0,255,255,255,  0,  0,255,255,255,255,255,255,255,255,  0,  0,  0,255,  6,  0,255,255,  0,  0,  0,255,255,  0,255, // õ,
          4,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  3,  2,  2,  0,  0,  3,255,  2,  2, 25,  0,  1,  0,  0,  0,  0,255,255,  0,255,255,255,  0,255,255,  0,255,255,255,  0,255,255,  0,255,  0,  0, 38,255,255,  0,  0,  0,255,255,  0,255, // ö,
          0,  0,  0,  3,  0,  4,  0,  0,  4,  0, 76, 18, 13,  8, 11, 19,  0,  0,  0,  9,  4, 27,  0,  3,  0,  0,  0,  0,255,255,255,  0,  0,255,255,  0,255,  0,255,  0,255,255,255,255,  0,255,255,255,255,  0,  0,255,  1,255,255,255,255,255, // ų,
          2,  0,  0,  9,  3,  1,  0,  0,  1,  0, 12,  8,  8,  3,  5,  0,  0,  1,  0,  9,  1, 12,  0,  1,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  1,  0,  0,  0,255,255,  0,  0,255,  0,  0,255,255,  0,255, // ū,
         93,  0,  2,  2,  0,  2,  0,  5,  2,  5,  1,  2, 43, 18,  7,  5,  0, 11,  0, 13, 48, 17,  0,  2,  0,  0,  0,  0,255,255,255,255,255,255,  0,255,255,  0,255,255,255,255,255,255,  0,255,255,  0,  0,255,255, 30,  0,  0,255,255,  0,255, // ü,
         47,  0, 17,  0,  0, 30, 11,  0,  1,  0,  3,  0,  0,  1,  3,  1,  2,  0,  0,  8,  0,  0, 16,  0,  0,  0,  2,  0,  0,255,255,  1,  1,  0,  0,  0,  0,  0,255,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255, // ž,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ·,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,  0,  0, // ¨,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0, // ®,
          0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,255,  0, // ©,
          0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,  0,  0,255, // ¬,
      //   ,  0,  a,  b,  c,  d,  e,  f,  g,  h,  i,  j,  k,  l,  m,  n,  o,  p,  q,  r,  s,  t,  u,  v,  w,  x,  y,  z,  ø,  ŗ,  ß,  ą,  į,  ā,  ä,  ę,  ē,  č,  é,  ė,  ģ,  ķ,  ī,  ļ,  š,  ņ,  ō,  õ,  ö,  ų,  ū,  ü,  ž,  ·,  ¨,  ®,  ©,  ¬,
    ],
    thai: [
                      0,  0, 11,  0,  0,  4,  0,  0, 25,  1,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  7,  4,  0,  2,  0, 35,  5,  2,  0,  0,  9,  0,  0, 15, 15,  8,  0,  7,  0,  5, 11,  1,  3,  0,  0,  7,  0,  0,  6,  0, 18,  1,  4, 14,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0, 17,  5,  0,  0, 28,  0,  0, //  ,
                      0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // 0,
                      0,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,  0,255,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // a,
          0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // €,
          0,  0,  0,255,  0,  0,  0,255,  0,255,255,  0,  0,255,  0,255,  0,  0,255,255,255,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,255,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,255,255,255,  0,  0,  0,  0,255,255,  0,  0,255, // ๐,
         15,  0,  0,255,  0,  2,  0,  0,  0,255,  0,  9,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  7,  2,  1,  0,  0,  0,  0,  0,  3,  5,  4,  0,  2,  0,  2,  0,  0,  1,  0,  0,  9,  0,  0,  4, 13, 16,  1,  8,  4,  2,  0,  2,  4,  0,  9,  3,  2,  0,  0,  0,  0,  2,  1,  1,  0,  0,  2,  0,  0, // ก,
          3,  0,  0,255,  0,  1,  0,  0,  0,255,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  6,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ข,
          0,  0,  0,255,255,  0,  0,  0,  0,255,255,  0,  0,255,  0,255,255,255,255,255,255,255,255,  0,255,  0,255,  0,255,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,255,  0,255,  0,255,255,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,255,  0,  0,255,255,  0,255,255, // ฃ,
         13,  0,  0,255,  0,  2,  0,  0,  1,255,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  1,  1,  2,  0,  0,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  8,  0,  1,  1,  0,  0,  1,  0,  0,  4,  1,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,255, // ค,
          0,255,  0,255,255,  0,  0,255,  0,255,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,  0,255,  0,255,255,255,255,255,255,255,  0,  0,  0,255,  0,255,255,255,255,255,255,255,  0,255,255,  0,  0,  0,  0,255,  0,255,255,255,  0,255,  0,  0,  0,255,255,255,  0,255,  0,255,255,255,  0,255,255, // ฅ,
          0,  0,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,255, // ฆ,
          0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  5,  0,  4,  0,  5,  0,  0,  1,  0,  0, 35,  0,  0,  0, 13, 22,  0,  8,  0,  3,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16,  8,  0,  0,  0,  0,  0, // ง,
          8,  0,  0,  0,  0,  1,  0,  0,  0,255,  0,  3,  1,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  4,  0,  2,  0,  0,  0,  0,  0,  0,  6,  0,  0,  1,  0,  0,  0,  2,  1,  1,  0,  0,  0,  0,255, // จ,
          0,  0,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255, // ฉ,
          6,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,  9,  0,  2,  1,  0,  0,  0,  0,  0,  5,  1,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ช,
          4,  0,  0,255,  0,  1,  0,  0,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ซ,
          0,255,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,255,255,  0,255,255, // ฌ,
          0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  1,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  3,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ญ,
          0,255,255,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,  0,  0,255,255,  0,  0,255, // ฎ,
          0,255,255,255,255,  0,255,255,  0,255,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  0,255,255,  0,255,255, // ฏ,
          0,  0,255,255,255,  0,  0,255,  0,255,255,  0,  0,255,  0,255,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  3,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0,255, // ฐ,
          0,255,255,255,255,  0,  0,255,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,255,255,  0,255,  0,  0,  0,  0,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,255,  0,255,  0,255,255, // ฑ,
          0,  0,  0,255,255,  0,255,255,  0,255,  0,  0,  0,255,255,255,255,255,  0,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,255,  0,255,255,  0,255,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,255,255,  0,255,255,  0,  1,  0,  0,  0,  0,255,255,  0,  0,  0,  0,255,  0,255,255,255,255,  0,255,  0,255,255,  0,  0,255, // ฒ,
          0,  0,255,255,  0,  0,  0,  0,  1,255,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,255,255,  1,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,  0,  0,255,  0,  0,255, // ณ,
          4,  0,  0,255,255,  1,  0,255,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  1,  0,  0,  0,  0,  0,  0,  1,  1,  2,  1,  1,255,  7,  0,  0,  2,  0,  0,  3,  0,  0,  1,  8,  3,  0,  7,  0,  0,  0,  3,  0,  0,  9,  1,  6,  0,  9,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0, // ด,
          7,  0,  0,255,  0,  2,  1,  0,  0,255,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  8,  0,  0,  0,  0,  0,  0,  0,  2,  1,  1,  0,  0,255,  0,  0,  0,  4,  1,  0,  2,  0,  0,  1,  5,  5,  0,  6,  1,  0,  0,  3,  0,  0,  4,  4,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0, // ต,
          1,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  2,  0,  0,255,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0, // ถ,
          8,  0,  0,255,  0,  1,  0,  0,  0,255,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0, 13,  3,  0,  0,  0,  0,  0,  0,  2,  1,  2,  0,  1,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  1,  4,  0,  5,  1,  0,  0,  2,  0,  0, 10,  1,  1,  0,  4,  0,  0,  0,  1,  1,  0,  0,  2,  0,255, // ท,
          1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  2,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255, // ธ,
          7,  0,  0,255,  0,  1,  1,  0,  3,  0,  0,  2,  1,  0,  3,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  1,  0,  1,  0,  4,  1,  0,  0,  0,  0,  0,  0,  3, 12,  1,  0,  2,255,  5,  1,  0,  3, 10,  0, 13,  0,  0,  1, 22, 19,  2, 11,  3,  0,  0,  2,  1,  0,  4,  1,  1, 18,  0,  0,  0, 15,  9, 11,  0,  0,  1,  0,  0, // น,
          6,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2,  4,  0,  0,  0,  1,  0,  0,  1,  1,  1,  0,  1,255,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2, 15,  2,  1,  2,  0,  0,  0,  1,  0,  0,  3,  4,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0, // บ,
          9,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  1,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  1,  0,  1,  0,  0,  0,  0,  1,  0, 19,  1,  1,  0,  5,  0,  0,  0,  1,  0,  0,  0,  0,  0,255, // ป,
          3,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ผ,
          0,  0,  0,255,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255, // ฝ,
         16,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,255,  0,  0,  0,  2,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  6,  0,  1,  1,  0,  0,  0,  0,  0,  7,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // พ,
          1,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,255,255, // ฟ,
          3,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ภ,
         10,  0,  0,255,  0,  2,  0,  0,  5,  0,  0,  3,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  1,  3,  7,  0,  0,255,  3,  0,  0,  6, 19,  0,  6,  0,  0,  2,  1, 17,  0,  4,  2,  0,  0,  2,  1,  0,  9,  3,  2,  0,  5,  0,  0,  1,  7,  2,  0,  0,  1,  0,  0, // ม,
          3,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  1,  0,  7,  0,  2,  0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  1,255,  5,  0,  0,  0,  0,  0,  8,  0,  0,  1,  5, 24,  0,  2, 20,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  6,  1,  0,  0,  0,  0,  0, // ย,
         11,  0,  0,255,  0, 15,  0,  0, 13,  0,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, 11,  0,  6,  2,  3,  6, 18,  0,  0, 13,  1,  0,  3,  2,  9,  0,  0,255,  3,  1,  0,  5,  4,  0, 10,  0,  0,  2,  0, 39,  1,  2,  1,  0,  0,  2,  1,  0, 10,  2,  4,  0,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0, // ร,
          0,  0,  0,255,255,  1,  0,255,  0,255,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  1,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,  0,255,255,  0,  0,255,  0,  0,  0,255,  0,255,255, // ฤ,
          3,  0,  0,255,  0,  8,  0,  0,  3,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  2,  6,  2,  0,  4,  2,  0,  0,  0,  0,  0,  1,255,  2,  0,  0,  0,  8,  0,  6,  0,  0,  0,  3,  6,  0,  4,  0,  0,  0,  2,  3,  0,  8, 15,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, // ล,
          0,255,255,255,255,  0,255,255,255,255,255,  0,255,255,255,255,255,  0,255,  0,255,255,255,255,  0,255,255,  0,255,  0,255,255,255,255,  0,255,255,255,  0,  0,255,255,  0,  0,  0,255,255,255,255,255,255,255,  0,255,  0,  0,  0,  0,255,255,255,255,255,  0,255,255,255,255,255,255,255,  0,255,255,255,  0,  0,255, // ฦ,
          7,  0,  0,255,  0,  3,  0,  0,  7,  0,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  5,  1,  0,  0,  0,  0,  0,  0,  8,  3,  4,  0,  1,255,  0,  0,  0,  1,  4,  0,  1,  0,  0,  2,  5, 10,  0,  2,  2,  0,  0,  0,  0,  0,  5,  0,  0,  0,  1,  0,  0,  0,  7,  6,  0,  0,  0,  0,  0, // ว,
         10,  0,  0,255,  0,  0,  0,255,  0,255,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255, // ศ,
          0,255,255,255,255,  5,  0,255,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,255,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  2,255,  1,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,255,255,  0,  0,255,  0,  0,  0,  0, // ษ,
         16,  0,  0,255,  0,  2,  0,  0,  0,255,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  6,  1,  0,  0,  0,  0,  0,  0,  4,  1,  3,  0,  1,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2,  2,  6,  0,  4,  1,  0,  0,  0,  0,  0,  5,  2,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  1,  0,  0, // ส,
         13,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  6,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  4,  1,  0,  0,  0,255,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,  5,  5,  0,  7,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0, // ห,
          0,255,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,255,255,255,  0,  0,255,255,  0,  0,  0,255,  0,255,  0,  0,255,255,255,255,  0,255,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,  0,  0,255,  0,255,255,  0,255, // ฬ,
         13,  0,  0,255,  0,  4, 12,  0,  2,  0,  0,  5,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  3,  3,  0,  2,  1,  6,  5,  1,  0,  0,  0,  0,  1,  3,  3,  6,255,  3,255,  2,  0,  0,  1,  0,  0,  5,  0,  0,  2,  0,  2,  1,  0,  1,  0, 14,  0,  0,  0,  7,  2,  3,  0,  0,  0,  0,  1, 24, 13,  0,  0,  1,  0,  0, // อ,
          1,  0,  0,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255, // ฮ,
          0,  0,  0,255,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,255,255,  0,255,255,255,  0,  0,  0,  0,255,  0,  0,  0,255, // ฯ,
          0,  0,  0,255,  0,  1,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  1,  0, 30,255, 14,255,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,255,255, // ะ,
          0,255,  0,255,255, 10,  1,  0,  2,  0,  0,  0,  7,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  7,  0,  5,  0,  9,  5,  2,  0,  0,  3,  0,  1,  4,  2, 16,255,  8,255, 13,  1,  1,  5,  1,  0,  5,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,255,  0,  0,  0,255,255,  0,  0,255, // ั,
          0,  0,  0,255,  0, 31,  3,  0,  2,  0,  0,  3,  8,  0,  9,  2,  0,  2,  0,  0,  1,  0,  0,  1,  3,  5,  2,  7,  1, 13,  5,  1,  0,  0,  2,  0, 11, 16,  9, 20,  0,  9,  0,  9,  3,  4,  8,  7,  1,  8,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,  0,  0, 20, 25,  0,  0,  0,  0,  0, // า,
          0,255,  0,255,255,  1,  0,  0,  2,  0,255,  0,  2,  0,  0,  0,255,  0,  0,255,255,255,255,  0,  0,  2,  0,  3,  0,  1,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  1,255,  0,  0,  0,  2,  0,255,  1,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  0,255,255,  0,255,255,  0,  0,  3,  0,  0,  0,255,255, // ำ,
          0,  0,  0,255,255,  6,  0,  0,  1,  0,  0,  0,  2,  0,  4,  2,  0,  1,  0,  0,  0,  0,  0,  0,  5,  9,  0,  1,  4,  6,  2,  2,  0,  0,  4,  0,  0,  4,  1, 11,  0,  6,255, 10,  1,  0,  5,  0,  0,  5,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,255,255,  0,255,255, // ิ,
          0,255,  0,255,  0,  3,  1,  0,  0,  0,  0,  0,  1,  0,  3,  3,  0,  1,  0,  0,  0,  0,  0,  0,  6,  1,  0, 28,  0,  7,  1,  5,  0,  0,  2,  0,  0, 11,  1, 12,  0,  7,255,  2,  0,  0,  4,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,255,  0,  0,  0,255,255,  0,255,255, // ี,
          0,255,  0,255,255,  0,  2,  0,  0,255,  0,  0,  1,  0,  0,  2,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  1,  0,  0,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,255,255,  0,255,255, // ึ,
          0,255,  0,255,255,  0,  0,  0,  2,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  3,  0,  0,  6,  0,  6,255,  2,255,  0,  0,  0,  1,  0,  0,  1,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,  0,255,255,  0,255,255, // ื,
          0,  0,  0,255,  0,  2,  0,  0,  2,  0,  0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  3,  0,  1,  0,  2,  3,  1,  0,  0,  1,  2,  0,  1,  3,  3,  0,  2,255,  0,  0,  0,  4,  0,  0,  2,  0,255,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,255,  0,  0,255,  0,255,255,255,255,  0,255,255, // ุ,
          0,  0,255,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  3,  1,  1,  0,  0,  0,  0,  0,  5,  0,  0,  0,  1,  9,  4,  3,  0,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,255,255,  0,255,255,  0,255,255,  0,  0,255,255,255,255,255,255,255,255,  0,255,255,255,255,  0,  0,255, // ู,
          0,  0,  0,255,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,255,  0,255,  0,  0,255,255,255,255,  0,255,  0,  0,255,  0,255,255,  0,255,255,255,255,  0,255,255, // ฺ,
         35,  0,  0,255,  0,  5,  0,  0,  0,  0,  0, 13,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  1,  0,  0,  0, 13,  2,  1,  0,  0,  0,  0,  0,  6,  4,  5,  0,  3,  0,  2,  0,  0,  2,  0,  0,  6,  0,  0, 12,  0,  7,  3,  3,  4,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  4,  0,  0,  4,  0,  0, // เ,
         18,  0,  0,255,  0,  2,  0,  0,  0,255,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  4,  1,  0,  0,  0,  0,  0,  0,  1,  1,  2,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  2,  0,255, // แ,
         10,  0,  0,255,  0,  0,  0,  0,  0,255,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  2,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,255, // โ,
          5,  0,  0,255,  0,  1,  0,  0,  0,255,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,255, // ใ,
          8,  0,  0,255,  0,  1,  0,255,  0,255,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  1,  0,  0,255,  0,  0,  0,  0,  1,  0,  0,  0,255,  1,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,255, // ไ,
          0,  0,255,255,255,  0,255,255,255,255,255,  0,  0,255,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,  0,255,255,255,  0,255,255,255,255,255,  0,255,  0,  0,255,255,  0,  0,255,  0,255,255,255,  0,255,255,255,255,  0,255,255,255,255,  0,255,255,255,255,  0,255,255,255,  0,255,255,  0,255,255, // ๅ,
          1,  0,  0,255,255,  0,255,255,  0,255,255,  0,255,  0,  0,  0,255,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,  0,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,255,255, // ๆ,
          0,255,  0,255,  0,  3,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,  0,  2,  0,  0,  0,  0,  0,  0, 13,  0,  0,  0,  0,  0,  0,  0,  1,255,  1,255,  0,  0,255,  0,  0,255,  1,  0,  0,  0,255,255,255,  0,  0,255,255,  0,  0,  0,  0,  0,255,255,255,255,255,  0,  0,  0,255,  0,  0,255,255, // ็,
          0,  0,  0,255,255,  3,  1,  0,  1,  0,  0,  0,  0,  0,  4,  0,  0,  1,  0,  0,  0,255,  0,  0,  0,  7,  0,  1,  0,  2,  1,  0,  1,  0,  0,  0,  0,  8,  3,  2,  0,  5,255,  6,  0,  0,  3,  3,  0,  0,  0,255,  0,  3,  0,  0,  4, 33,  4, 14,  4, 11,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255, // ่,
          0,  0,  0,255,  0,  2,  6,  0,  1,  0,  0,  0,  4,  0,  4,  1,  0,  0,  0,255,255,255,255,255, 12,  4,  0,  1,  0,  6,  1,  0,  0,  0,  0,  0,  0,  1,  0,  4,255,  3,255,  1,  0,  0,  0,  5,255,  2,  0,  0,  0, 11,  0,  0,  1,  4,  2,  2,  0,  7,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,255,255,  0,  0,255, // ้,
          0,255,255,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,255,  0,  0,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,  0,  0,255,255,  0,255,255,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,255,  0,255,  0,  0,255,255,255,255, // ๊,
          0,255,  0,255,255,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,255,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,255,  0,  0,255,255,255,  0,255,255, // ๋,
          0,255,  0,255,  0,  0,  0,  0,  2,255,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  2,  2,  3,  0,  1,  1,  3,  0,  1,  0,255,  0,  0,  0,  0,  5, 20,255,  3,255,  1,  1,  0,  3,  1,  0,  0,  0,  0,  0,  0,  0,255,  1,  0,  0,  0,  0,  0,  0,  0,  0,255,255,255,255,255,  0,255,255,  0,  0,  0,  0,255, // ์,
          0,  0,  0,255,255,  0,  0,255,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,  0,  0,  0,  0,  0,255,  0,  0,  0,  0,  0,  0,255,  0,255,255,255,  0,  0,255,  0,  0,255,255,255,255,255,255,255,  0,  0,255,  0,  0,255, // ํ,
          0,255,255,255,255,  0,  0,255,255,255,255,255,  0,255,255,255,255,  0,255,255,  0,255,255,  0,  0,  0,255,  0,255,  0,255,  0,  0,  0,  0,255,255,  0,  0,  0,255,  0,255,  0,255,255,  0,  0,  0,  0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255, // ๎,
      //   ,  0,  a,  €,  ๐,  ก,  ข,  ฃ,  ค,  ฅ,  ฆ,  ง,  จ,  ฉ,  ช,  ซ,  ฌ,  ญ,  ฎ,  ฏ,  ฐ,  ฑ,  ฒ,  ณ,  ด,  ต,  ถ,  ท,  ธ,  น,  บ,  ป,  ผ,  ฝ,  พ,  ฟ,  ภ,  ม,  ย,  ร,  ฤ,  ล,  ฦ,  ว,  ศ,  ษ,  ส,  ห,  ฬ,  อ,  ฮ,  ฯ,  ะ,  ั,  า,  ำ,  ิ,  ี,  ึ,  ื,  ุ,  ู,  ฺ,  เ,  แ,  โ,  ใ,  ไ,  ๅ,  ๆ,  ็,  ่,  ้,  ๊,  ๋,  ์,  ํ,  ๎,
    ],
};

const VIETNAMESE_ASCII: usize = 28;
const VIETNAMESE_NON_ASCII: usize = 31;
const CENTRAL_ASCII: usize = 28;
const CENTRAL_NON_ASCII: usize = 45;
const CYRILLIC_ASCII: usize = 3;
const CYRILLIC_NON_ASCII: usize = 52;
const WESTERN_ASCII: usize = 28;
const WESTERN_NON_ASCII: usize = 39;
const ICELANDIC_ASCII: usize = 28;
const ICELANDIC_NON_ASCII: usize = 18;
const GREEK_ASCII: usize = 3;
const GREEK_NON_ASCII: usize = 40;
const TURKISH_ASCII: usize = 27;
const TURKISH_NON_ASCII: usize = 19;
const HEBREW_ASCII: usize = 4;
const HEBREW_NON_ASCII: usize = 55;
const ARABIC_ASCII: usize = 3;
const ARABIC_NON_ASCII: usize = 65;
const BALTIC_ASCII: usize = 28;
const BALTIC_NON_ASCII: usize = 30;
const THAI_ASCII: usize = 3;
const THAI_NON_ASCII: usize = 75;
#[inline(always)]
fn compute_index(
    x: usize,
    y: usize,
    ascii_classes: usize,
    non_ascii_classes: usize,
) -> Option<usize> {
    if x == 0 && y == 0 {
        return None;
    }
    if x < ascii_classes && y < ascii_classes {
        return None;
    }
    if y >= ascii_classes {
        return Some(
            (ascii_classes * non_ascii_classes)
                + (ascii_classes + non_ascii_classes) * (y - ascii_classes)
                + x,
        );
    }
    Some(y * non_ascii_classes + x - ascii_classes)
}

pub struct SingleByteData {
    pub encoding: &'static Encoding,
    lower: &'static [u8; 128],
    upper: &'static [u8; 128],
    probabilities: &'static [u8],
    ascii: usize,
    non_ascii: usize,
    letter_boundary: u8,
}

impl SingleByteData {
    #[inline(always)]
    pub fn classify(&'static self, byte: u8) -> u8 {
        let high = byte >> 7;
        let low = byte & 0x7F;
        if high == 0u8 {
            self.lower[usize::from(low)]
        } else {
            self.upper[usize::from(low)]
        }
    }

    #[inline(always)]
    pub fn is_latin_alphabetic(&'static self, byte: u8) -> bool {
        return byte >= 2 && byte < self.letter_boundary;
    }

    #[inline(always)]
    pub fn is_non_latin_alphabetic(&'static self, byte: u8) -> bool {
        return byte >= self.letter_boundary;
    }

    #[inline(always)]
    pub fn score(&'static self, current_class: u8, previous_class: u8) -> i64 {
        if let Some(index) = compute_index(
            usize::from(previous_class),
            usize::from(current_class),
            self.ascii,
            self.non_ascii,
        ) {
            let b = self.probabilities[index];
            if b == 255 {
                IMPLAUSIBILITY_PENALTY
            } else {
                i64::from(b)
            }
        } else {
            0
        }
    }
}

impl PartialEq for SingleByteData {
    #[inline]
    fn eq(&self, other: &SingleByteData) -> bool {
        (self as *const SingleByteData) == (other as *const SingleByteData)
    }
}

pub static SINGLE_BYTE_DATA: [SingleByteData; 20] = [
    SingleByteData {
        encoding: &WINDOWS_1258_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.windows_1258,
        probabilities: &DETECTOR_DATA.vietnamese,
        ascii: VIETNAMESE_ASCII,
        non_ascii: VIETNAMESE_NON_ASCII,
        letter_boundary: 29,
    },
    SingleByteData {
        encoding: &WINDOWS_1250_INIT,
        lower: &DETECTOR_DATA.latin_ascii,
        upper: &DETECTOR_DATA.windows_1250,
        probabilities: &DETECTOR_DATA.central,
        ascii: CENTRAL_ASCII,
        non_ascii: CENTRAL_NON_ASCII,
        letter_boundary: 69,
    },
    SingleByteData {
        encoding: &ISO_8859_2_INIT,
        lower: &DETECTOR_DATA.latin_ascii,
        upper: &DETECTOR_DATA.iso_8859_2,
        probabilities: &DETECTOR_DATA.central,
        ascii: CENTRAL_ASCII,
        non_ascii: CENTRAL_NON_ASCII,
        letter_boundary: 69,
    },
    SingleByteData {
        encoding: &WINDOWS_1251_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.windows_1251,
        probabilities: &DETECTOR_DATA.cyrillic,
        ascii: CYRILLIC_ASCII,
        non_ascii: CYRILLIC_NON_ASCII,
        letter_boundary: 8,
    },
    SingleByteData {
        encoding: &KOI8_U_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.koi8_u,
        probabilities: &DETECTOR_DATA.cyrillic,
        ascii: CYRILLIC_ASCII,
        non_ascii: CYRILLIC_NON_ASCII,
        letter_boundary: 8,
    },
    SingleByteData {
        encoding: &ISO_8859_5_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.iso_8859_5,
        probabilities: &DETECTOR_DATA.cyrillic,
        ascii: CYRILLIC_ASCII,
        non_ascii: CYRILLIC_NON_ASCII,
        letter_boundary: 8,
    },
    SingleByteData {
        encoding: &IBM866_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.ibm866,
        probabilities: &DETECTOR_DATA.cyrillic,
        ascii: CYRILLIC_ASCII,
        non_ascii: CYRILLIC_NON_ASCII,
        letter_boundary: 8,
    },
    SingleByteData {
        encoding: &WINDOWS_1252_INIT,
        lower: &DETECTOR_DATA.latin_ascii,
        upper: &DETECTOR_DATA.windows_1252,
        probabilities: &DETECTOR_DATA.western,
        ascii: WESTERN_ASCII,
        non_ascii: WESTERN_NON_ASCII,
        letter_boundary: 61,
    },
    SingleByteData {
        encoding: &WINDOWS_1252_INIT,
        lower: &DETECTOR_DATA.latin_ascii,
        upper: &DETECTOR_DATA.windows_1252_icelandic,
        probabilities: &DETECTOR_DATA.icelandic,
        ascii: ICELANDIC_ASCII,
        non_ascii: ICELANDIC_NON_ASCII,
        letter_boundary: 41,
    },
    SingleByteData {
        encoding: &WINDOWS_1253_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.windows_1253,
        probabilities: &DETECTOR_DATA.greek,
        ascii: GREEK_ASCII,
        non_ascii: GREEK_NON_ASCII,
        letter_boundary: 7,
    },
    SingleByteData {
        encoding: &ISO_8859_7_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.iso_8859_7,
        probabilities: &DETECTOR_DATA.greek,
        ascii: GREEK_ASCII,
        non_ascii: GREEK_NON_ASCII,
        letter_boundary: 7,
    },
    SingleByteData {
        encoding: &WINDOWS_1254_INIT,
        lower: &DETECTOR_DATA.turkish_ascii,
        upper: &DETECTOR_DATA.windows_1254,
        probabilities: &DETECTOR_DATA.turkish,
        ascii: TURKISH_ASCII,
        non_ascii: TURKISH_NON_ASCII,
        letter_boundary: 41,
    },
    SingleByteData {
        encoding: &WINDOWS_1255_INIT,
        lower: &DETECTOR_DATA.hebrew_ascii,
        upper: &DETECTOR_DATA.windows_1255,
        probabilities: &DETECTOR_DATA.hebrew,
        ascii: HEBREW_ASCII,
        non_ascii: HEBREW_NON_ASCII,
        letter_boundary: 11,
    },
    SingleByteData {
        encoding: &ISO_8859_8_INIT,
        lower: &DETECTOR_DATA.hebrew_ascii,
        upper: &DETECTOR_DATA.iso_8859_8,
        probabilities: &DETECTOR_DATA.hebrew,
        ascii: HEBREW_ASCII,
        non_ascii: HEBREW_NON_ASCII,
        letter_boundary: 11,
    },
    SingleByteData {
        encoding: &WINDOWS_1256_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.windows_1256,
        probabilities: &DETECTOR_DATA.arabic,
        ascii: ARABIC_ASCII,
        non_ascii: ARABIC_NON_ASCII,
        letter_boundary: 11,
    },
    SingleByteData {
        encoding: &ISO_8859_6_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.iso_8859_6,
        probabilities: &DETECTOR_DATA.arabic,
        ascii: ARABIC_ASCII,
        non_ascii: ARABIC_NON_ASCII,
        letter_boundary: 11,
    },
    SingleByteData {
        encoding: &WINDOWS_1257_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.windows_1257,
        probabilities: &DETECTOR_DATA.baltic,
        ascii: BALTIC_ASCII,
        non_ascii: BALTIC_NON_ASCII,
        letter_boundary: 53,
    },
    SingleByteData {
        encoding: &ISO_8859_13_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.iso_8859_13,
        probabilities: &DETECTOR_DATA.baltic,
        ascii: BALTIC_ASCII,
        non_ascii: BALTIC_NON_ASCII,
        letter_boundary: 53,
    },
    SingleByteData {
        encoding: &ISO_8859_4_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.iso_8859_4,
        probabilities: &DETECTOR_DATA.baltic,
        ascii: BALTIC_ASCII,
        non_ascii: BALTIC_NON_ASCII,
        letter_boundary: 53,
    },
    SingleByteData {
        encoding: &WINDOWS_874_INIT,
        lower: &DETECTOR_DATA.non_latin_ascii,
        upper: &DETECTOR_DATA.windows_874,
        probabilities: &DETECTOR_DATA.thai,
        ascii: THAI_ASCII,
        non_ascii: THAI_NON_ASCII,
        letter_boundary: 5,
    },
];

pub const WINDOWS_1258_INDEX: usize = 0;
pub const WINDOWS_1250_INDEX: usize = 1;
pub const ISO_8859_2_INDEX: usize = 2;
pub const WINDOWS_1251_INDEX: usize = 3;
pub const KOI8_U_INDEX: usize = 4;
pub const ISO_8859_5_INDEX: usize = 5;
pub const IBM866_INDEX: usize = 6;
pub const WINDOWS_1252_INDEX: usize = 7;
pub const WINDOWS_1252_ICELANDIC_INDEX: usize = 8;
pub const WINDOWS_1253_INDEX: usize = 9;
pub const ISO_8859_7_INDEX: usize = 10;
pub const WINDOWS_1254_INDEX: usize = 11;
pub const WINDOWS_1255_INDEX: usize = 12;
pub const ISO_8859_8_INDEX: usize = 13;
pub const WINDOWS_1256_INDEX: usize = 14;
pub const ISO_8859_6_INDEX: usize = 15;
pub const WINDOWS_1257_INDEX: usize = 16;
pub const ISO_8859_13_INDEX: usize = 17;
pub const ISO_8859_4_INDEX: usize = 18;
pub const WINDOWS_874_INDEX: usize = 19;
