# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.
import json
import os
import re
import signal
import tempfile
import threading

from mozlog import get_proxy_logger
from mozperftest.layers import Layer
from mozperftest.utils import download_file, install_package
from mozprocess import ProcessHandler


LOG = get_proxy_logger(component="proxy")
HERE = os.path.dirname(__file__)


class OutputHandler(object):
    def __init__(self):
        self.proc = None
        self.port = None
        self.port_event = threading.Event()

    def __call__(self, line):
        line = line.strip()
        if not line:
            return
        line = line.decode("utf-8", errors="replace")
        try:
            data = json.loads(line)
        except ValueError:
            self.process_output(line)
            return

        if isinstance(data, dict) and "action" in data:
            # Retrieve the port number for the proxy server from the logs of
            # our subprocess.
            m = re.match(r"Proxy running on port (\d+)", data.get("message", ""))
            if m:
                self.port = int(m.group(1))
                self.port_event.set()
            LOG.log_raw(data)
        else:
            self.process_output(json.dumps(data))

    def finished(self):
        self.port_event.set()

    def process_output(self, line):
        if self.proc is None:
            LOG.process_output(line)
        else:
            LOG.process_output(self.proc.pid, line)

    def wait_for_port(self):
        self.port_event.wait()
        return self.port


class ProxyRunner(Layer):
    """Use a proxy"""

    name = "proxy"
    activated = False

    arguments = {
        "record": {
            "type": str,
            "default": None,
            "help": "Generate a recording of the network requests during this test.",
        },
        "replay": {
            "type": str,
            "default": None,
            "help": "A generated recording to play back during this test.",
        },
    }

    def __init__(self, env, mach_cmd):
        super(ProxyRunner, self).__init__(env, mach_cmd)
        self.proxy = None
        self.tmpdir = None

    def setup(self):
        # Install mozproxy and its vendored deps.
        mozbase = os.path.join(self.mach_cmd.topsrcdir, "testing", "mozbase")
        mozproxy_deps = ["mozinfo", "mozlog", "mozproxy"]
        for i in mozproxy_deps:
            install_package(self.mach_cmd.virtualenv_manager, os.path.join(mozbase, i))

    def run(self, metadata):
        self.metadata = metadata
        replay_file = self.get_arg("replay")
        if replay_file is not None and replay_file.startswith("http"):
            self.tmpdir = tempfile.TemporaryDirectory()
            target = os.path.join(self.tmpdir.name, "recording.zip")
            self.info("Downloading %s" % replay_file)
            download_file(replay_file, target)
            replay_file = target

        self.info("Setting up the proxy")
        command = [
            self.mach_cmd.virtualenv_manager.python_path,
            "-m",
            "mozproxy.driver",
            "--local",
            "--binary=" + self.mach_cmd.get_binary_path(),
            "--topsrcdir=" + self.mach_cmd.topsrcdir,
            "--objdir=" + self.mach_cmd.topobjdir,
        ]
        if self.get_arg("record"):
            command.extend(["--record", self.get_arg("record")])
        elif replay_file:
            command.append(replay_file)
        else:
            command.append(os.path.join(HERE, "example.zip"))
        print(" ".join(command))
        self.output_handler = OutputHandler()
        self.proxy = ProcessHandler(
            command,
            processOutputLine=self.output_handler,
            onFinish=self.output_handler.finished,
        )
        self.output_handler.proc = self.proxy
        self.proxy.run()

        # Wait until we've retrieved the proxy server's port number so we can
        # configure the browser properly.
        port = self.output_handler.wait_for_port()
        if port is None:
            raise ValueError("Unable to retrieve the port number from mozproxy")
        self.info("Received port number %s from mozproxy" % port)

        prefs = {
            "network.proxy.type": 1,
            "network.proxy.http": "localhost",
            "network.proxy.http_port": port,
            "network.proxy.ssl": "localhost",
            "network.proxy.ssl_port": port,
            "network.proxy.no_proxies_on": "localhost",
        }
        browser_prefs = metadata.get_options("browser_prefs")
        browser_prefs.update(prefs)
        return metadata

    def teardown(self):
        err = None
        if self.proxy is not None:
            returncode = self.proxy.wait(0)
            if returncode is not None:
                err = ValueError(
                    "mozproxy terminated early with return code %d" % returncode
                )
            else:
                kill_signal = getattr(signal, "CTRL_BREAK_EVENT", signal.SIGINT)
                os.kill(self.proxy.pid, kill_signal)
                self.proxy.wait()
            self.proxy = None
        if self.tmpdir is not None:
            self.tmpdir.cleanup()
            self.tmpdir = None

        if err:
            raise err
